-----------------------------------------------------------
-- SmartRes
-- 
-- Automated and coordinated raid resurrection. Listen for
-- resurrects from people with SmartRes, CTRA, oRA2, 
-- CastCommLib and nearby people. Displays info with 
-- CandyBars.
--
-- Original addon by Maia
-- Work done by Kyahx
-- Continued by poull
-- WotLK changes by Myrroddin, bug fixes by Zidomo
-----------------------------------------------------------

-----------------------------------------------------------
-- Libraries
-----------------------------------------------------------
local L =  AceLibrary("AceLocale-2.2"):new("SmartRes")
local DD = AceLibrary("Dewdrop-2.0")
local SM = AceLibrary("SharedMedia-1.0")
local P =  AceLibrary("PaintChips-2.0")

-----------------------------------------------------------
-- Addon declaration
-----------------------------------------------------------
SmartRes = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "AceConsole-2.0", "AceHook-2.1", "CandyBar-2.0", "FuBarPlugin-2.0")
local SmartRes = SmartRes

-----------------------------------------------------------
-- FuBar Plugin
-----------------------------------------------------------
SmartRes.hasIcon = "Interface\\Icons\\Spell_Holy_Resurrection"  
SmartRes.defaultMinimapPosition = 180
SmartRes.cannotDetachTooltip = true
SmartRes.hasNoColor = true
SmartRes.clickableTooltip = true
SmartRes.hideWithoutStandby = true
SmartRes.independentProfile = true

-----------------------------------------------------------
-- Initialization
-----------------------------------------------------------
function SmartRes:OnInitialize()
	-- Local lists
	self.ResTable = {}
	self.RessingTable = {}
	self.PossibleRessingTable = {}
	self.RessedTable = {}
	self.BlackListTable = {}
	self.Casting = nil
	self.Combat = false

	-- Prepare spells
	local ResSpells = {
		PRIEST = GetSpellInfo(2006), -- Resurrection
    SHAMAN = GetSpellInfo(2008), -- Ancestral Spirit
    PALADIN = GetSpellInfo(7328), -- Redemption
    DRUID = GetSpellInfo(50769) -- Revive
	}
	self.ResSpellIcons = {
		PRIEST = select(3, GetSpellInfo(2006)), -- Resurection
    SHAMAN = select(3, GetSpellInfo(2008)), -- Ancestral Spirit
    PALADIN = select(3, GetSpellInfo(7328)), -- Redemption
    DRUID = select(3, GetSpellInfo(50769)) -- Revive
	}
	self.ResSpells = ResSpells
	self.PlayerClass = select(2, UnitClass("player"))
	self.PlayerSpell = ResSpells[self.PlayerClass]

	-- Secure button for casting spells
	local resbutton = CreateFrame("Button", "SmartResButton", UIParent, "SecureActionButtonTemplate")
        resbutton:SetAttribute("type", "spell")
        resbutton:SetScript("PreClick", function() self:Resurrect() end);
	self.ResButton = resbutton

	self:InitializeOptions() -- See SmartResOptions.lua
end

function SmartRes:OnEnable()
	if self.PlayerSpell then
		self:BindKeys()
		self:RegisterNoCombatEvents()
		self:RegisterCombatEvents()
	end
end

function SmartRes:OnDisable()
	if self.PlayerSpell then
		self:UnregisterAllEvents()
		self:UnbindKeys()
	end
end
-----------------------------------------------------------
-- Debug
-----------------------------------------------------------
function SmartRes:Debug(text)
	self:Print("|cff0099CCDebug - "..text.."|r")
end

-----------------------------------------------------------
-- Key and event initialization
-----------------------------------------------------------
function SmartRes:BindKeys()
	SetOverrideBindingClick(self.ResButton, false, self.db.profile.autokey, "SmartResButton")
	SetOverrideBindingSpell(self.ResButton, false, self.db.profile.normalkey, self.PlayerSpell)
end

function SmartRes:UnbindKeys()
	ClearOverrideBindings(self.ResButton)
end

function SmartRes:RegisterNoCombatEvents()
	if not self:IsEventRegistered("UNIT_HEALTH") then
		self:RegisterBucketEvent("UNIT_HEALTH", 0.1)
		self:RegisterEvent("UNIT_SPELLCAST_SENT")
		self:RegisterEvent("UNIT_SPELLCAST_START")
		self:RegisterEvent("UNIT_SPELLCAST_INTERRUPTED")
		self:RegisterEvent("UNIT_SPELLCAST_FAILED")
		self:RegisterEvent("UNIT_SPELLCAST_STOP")
		self:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
		self:RegisterEvent("RESURRECT_REQUEST")
		self:HookScript(WorldFrame, "OnMouseDown", "WorldFrameOnMouseDown")
	end
end

function SmartRes:RegisterCombatEvents()
        self:RegisterEvent("PLAYER_REGEN_DISABLED");
        self:RegisterEvent("PLAYER_REGEN_ENABLED");
	self:RegisterEvent("CHAT_MSG_ADDON")
end

function SmartRes:UnregisterNoCombatEvents()
	if self:IsEventRegistered("UNIT_HEALTH") then
		self:UnregisterBucketEvent("UNIT_HEALTH")
		self:UnregisterEvent("UNIT_SPELLCAST_SENT")
		self:UnregisterEvent("UNIT_SPELLCAST_START")
		self:UnregisterEvent("UNIT_SPELLCAST_INTERRUPTED")
		self:UnregisterEvent("UNIT_SPELLCAST_FAILED")
		self:UnregisterEvent("UNIT_SPELLCAST_STOP")
		self:UnregisterEvent("UNIT_SPELLCAST_SUCCEEDED")
		self:UnregisterEvent("RESURRECT_REQUEST")
		self:Unhook(WorldFrame, "OnMouseDown")
	end
end

-----------------------------------------------------------
-- Events
-----------------------------------------------------------
function SmartRes:PLAYER_REGEN_DISABLED()
	-- Unregister to avoid using resources while in combat
	self:UnregisterNoCombatEvents()
	self.Combat = true
end

function SmartRes:PLAYER_REGEN_ENABLED()
	-- Register again after combat
	self:RegisterNoCombatEvents()
	self.Combat = false

	-- Check for spells being cast since we don't listen in combat
	self.Casting = true and UnitCastingInfo("player")
end

function SmartRes:UNIT_HEALTH(units)
	-- Used to track people without addon being resurrected by
	-- players with addon and who dies before RessedTable runs out
	for unit in pairs(units) do
		local dead = UnitIsDead(unit)
		local name = UnitName(unit)
		if not dead and self.RessedTable[name] then
			self.RessedTable[name] = nil
			self.BlackListTable[name] = nil
			self:Alive(name)
		end
	end
end

--[[ Event order

Attempting to cast resurrection while moving:
UNIT_SPELLCAST_FAILED (two times in raids)

Attempting to cast resurrection with target out of line of sight:
UNIT_SPELLCAST_SENT
UNIT_SPELLCAST_FAILED (two times in raids)

Casting resurrection which gets interruped:
UNIT_SPELLCAST_SENT
UNIT_SPELLCAST_START (two times in raids)
UNIT_SPELLCAST_STOP (two times in raids)
UNIT_SPELLCAST_INTERRUPTED (two times in raids)
UNIT_SPELLCAST_STOP (two times in raids)

Casting resurrection successfully:
UNIT_SPELLCAST_SENT
UNIT_SPELLCAST_START (two times in raids)
UNIT_SPELLCAST_SUCCEEDED
UNIT_SPELLCAST_STOP (two times in raids)

Observing resurrection which gets interrupted:
UNIT_SPELLCAST_START (two times in raids)
UNIT_SPELLCAST_STOP (two times in raids)

Observing successfull resurrection:
UNIT_SPELLCAST_START (two times in raids)

]]

function SmartRes:UNIT_SPELLCAST_SENT(unit, spell, rank, target)
	--self:Print("UNIT_SPELLCAST_SENT")
	self.Casting = true
	if spell == self.PlayerSpell then
		if not target or target == "" or target == "Unknown" then
			-- Players who have released and are targeted manually
			-- appear as target == "".
			target = self.MouseDownTarget
		end
		self.Target = target
		if target then
			self:SendAddonMessage("RES " .. target)
		end
	else
		self.Target = nil
	end
end

function SmartRes:UNIT_SPELLCAST_START(unit)
	--self:Print("UNIT_SPELLCAST_START")
	local _, unitclass = UnitClass(unit);
	local spell, _, _, _, starttime, endtime = UnitCastingInfo(unit)
	if self.ResSpells[unitclass] == spell then
		if unit == "player" then
			self:SendAddonMessage("RESINFO", true)
		elseif not UnitIsUnit(unit, "player") then
			local ressername = UnitName(unit)
			local unittarget = unit .. "target"
			if UnitExists(unittarget) and UnitIsDead(unittarget) and not self.RessingTable[ressername] then
				local targetname = UnitName(unittarget)
				self.RessingTable[ressername] = targetname
				local castingtime = (endtime - starttime) / 1000
				--self:Debug(string.format("Resurrect of %s from %s via %s", targetname, ressername, "UNIT_SPELLCAST_START"))
				self:ResurrectStart(ressername, targetname, castingtime)
			end
		end
	end
end

function SmartRes:UNIT_SPELLCAST_INTERRUPTED(unit)
	--self:Print("UNIT_SPELLCAST_INTERRUPTED")
	if unit == "player" and self.Target then
		self.BlackListTable[self.Target] = (self.BlackListTable[self.Target] or 0) + 10
	end
end

function SmartRes:UNIT_SPELLCAST_FAILED(unit)
	--self:Print("UNIT_SPELLCAST_FAILED")
	if UnitIsUnit(unit, "player") and self.Target then
		--self:Print(format(L["Failed to res %s"], self.Target))
		if not UnitIsAFK(unit) then
			-- Spells fail for two reasons, line of sight and sitting. People who
			-- sit down are often afk so we don't punish that case.
			self.BlackListTable[self.Target] = 25
		end
		self:UNIT_SPELLCAST_STOP(unit)
	end
end

function SmartRes:UNIT_SPELLCAST_SUCCEEDED(unit, spell, rank)
	--self:Print("UNIT_SPELLCAST_SUCCEEDED")
	local target = self.Target
	if unit == "player" and spell == self.PlayerSpell and target then
		self.RessedTable[self.Target] = GetTime()
		self.BlackListTable[self.Target] = nil
		self:Resurrected(player)
		self:SendAddonMessage("RESSED " .. target, true)
	end
end

function SmartRes:UNIT_SPELLCAST_STOP(unit)
	--self:Print("UNIT_SPELLCAST_STOP")
	if unit == "player" and self.Casting then
		self.Casting = nil
		if self.Target then
			self:SendAddonMessage("RESNO")
		end
	else
		local name = UnitName(unit)
		if self.RessingTable[name] then
			self.RessingTable[name] = nil
			self:ResurrectStop(name)
		end
	end
end

function SmartRes:RESURRECT_REQUEST()
	if not self.RessedTable[UnitName("player")] then
		self:SendAddonMessage("RESSED")
	end
end

function SmartRes:PLAYER_DEAD()
	if HasSoulstone() then
		self:SendAddonMessage("CANRES")
	end
end

function SmartRes:CHAT_MSG_ADDON(prefix, msg, type, author)
	if type ~= "RAID" and type ~= "PARTY" then return end
	if prefix ~= "CTRA" and prefix ~= "oRA" and prefix ~= "SmartRes" and prefix ~= "CastCommLib" then return end
	--self:Print(prefix .. ": " .. msg .. " (" .. type .. " - " .. author .. ")")

	for cmd, val in msg:gmatch("(%a+)%s?([^#]*)") do
		if val == "" then val = nil end

		if cmd == "RES" and not self.RessingTable[author] and val then
			self.RessingTable[author] = val
			--self:Debug(string.format("Resurrect of %s from %s via %s", val, author, "SmartRes"))
			self:ResurrectStart(author, val)
		end

		if cmd == "C" and not self.RessingTable[author] and val then
			local spell, _, _, _, starttime, endtime = UnitCastingInfo(author)
			local authorclass = select(2, UnitClass(author))
			if spell and self.ResSpells[authorclass] == spell then
				self.RessingTable[author] = val
				local castingtime = (endtime - starttime) / 1000
				--self:Debug(string.format("Resurrect of %s from %s via %s", val, author, "CastCommLib"))
				self:ResurrectStart(author, val, castingtime)
			end
		end

		if cmd == "RESINFO" then
			if self.RessingTable[author] then
				local _, _, _, _, starttime, endtime = UnitCastingInfo(author)
				-- Check in case cast is cancelled
				if starttime and endtime then
					local castingtime = (endtime - starttime) / 1000
					self:ResurrectUpdateTime(author, castingtime)
				end
			end
		end

		if cmd == "RESSED" or (cmd == "S" and self.RessingTable[author]) then
			local player = val or author
			local now = GetTime()
			local ressed = self.RessedTable[player]
			-- Register res if not already done within the last 2 sec
			if not ressed or now - ressed > 2 then
				self.RessedTable[player] = now
				self.BlackListTable[player] = nil
				self:Resurrected(player)
			end
		end

		if cmd == "RESNO" or ((cmd == "I" or cmd == "S") and self.RessingTable[author]) then
			self.RessingTable[author] = nil
			self:ResurrectStop(author)
		end

		if cmd == "NORESSED" then
			if self.RessedTable[author] then
				-- Add 2 sec delay where player is still flagged as ressed
				-- to account for time between resurrect dialog is closed
				-- and the player is actually alive
				self.RessedTable[author] = GetTime() - 58
			end
			self:NotResurrected(author)
		end

		if cmd == "CANRES" then
			-- Lower priority for people who can use soulstone
			self.BlackListTable[author] = 25
			self:Soulstone(author)
		end
	end

end

-- Fix for UNIT_SPELLCAST_SENT target being empty (from oRA2)
function SmartRes:WorldFrameOnMouseDown(...)
	if GameTooltipTextLeft1:IsVisible() then
		local target = select(3, GameTooltipTextLeft1:GetText():find("^Corpse of (.+)$"))
		if target then
			self.MouseDownTarget = target
		end
	end
	self.hooks[WorldFrame]["OnMouseDown"](...)
end

function SmartRes:SendAddonMessage(msg, smartres)
	if smartres then
		SendAddonMessage("SmartRes", msg, "RAID")
	else
		if not (oRA and oRA.HasModule(oRA, "ParticipantPassive") and oRA.IsModuleActive(oRA, "ParticipantPassive") or CT_RA_Stats) then
			-- Only send when oRA2/CTRA is inactive
			SendAddonMessage("CTRA", msg, "RAID")
		end
	end
end

-----------------------------------------------------------
-- Functions triggered from events
-----------------------------------------------------------
function SmartRes:ResurrectStart(player, target, castingtime)
	if UnitIsUnit(player, "player") then
		self:OutputTextNotifications(target)
	end

	self:CreateBar(player, target, castingtime or 10, self:CollidingRes(player, target))
end

-- Triggered when new info on casting time is available
function SmartRes:ResurrectUpdateTime(player, castingtime)
	self:SetCandyBarTime(player, castingtime)
end

function SmartRes:ResurrectStop(player)
	self:StopCandyBar(player)
end

function SmartRes:Resurrected(player)

end

function SmartRes:Alive(player)

end

function SmartRes:NotResurrected(player)

end

function SmartRes:Soulstone(player)

end

-----------------------------------------------------------
-- Choosing who to res
-----------------------------------------------------------
function SmartRes:Resurrect()
	local resbutton = self.ResButton

	if self.Casting then return end

	if GetNumPartyMembers() == 0 and not UnitInRaid("player") then
		self:Print(L["You're not in a group."])
		return
	end

	local usable, nomana = IsUsableSpell(self.PlayerSpell)
	if not usable then
		if nomana then
			self:Print(L["Not enough mana to res."])
		else
			if UnitAffectingCombat("player") then
				self:Print(L["Can't res while in combat."])
			else
				self:Print(L["Can't res right now."])
			end
		end
		return
	end
	resbutton:SetAttribute("unit", nil)
        resbutton:SetAttribute("spell", nil)

	self.ResTable = {}
	if UnitInRaid("player") then
		--self:Debug("Building Raid ResTable")
		self:BuildResTable("raid")
	else
		--self:Debug("Building Party ResTable")
		self:BuildResTable("party")
	end
	if self.NumCorpses == 0 then
		self:Print(L["Coudnt't find any dead (non-released) units."])
		return
	end
	if self.NumCorpses - self.NumCorpsesRessed == 0 then
		self:Print(L["All dead units are already being ressed."])
		return
	end
	table.sort(
		self.ResTable, 
		function(val1, val2)
			return val1[2] < val2[2]
		end
	)

	self.corpse = self:ChooseDeadUnitInRange()
	if self.corpse then
	        resbutton:SetAttribute("unit", self.corpse)
	        resbutton:SetAttribute("spell", self.PlayerSpell)
	else
		self:Print(L["No one in range to res."])
	end
end

function SmartRes:BuildResTable(type)
	local x
	if type == "raid" then x = GetNumRaidMembers()
	elseif type == "party" then x = GetNumPartyMembers()
	end
	self.NumCorpses = 0
	self.NumCorpsesRessed = 0
	for i = 1, x do
		typei = type .. i
		if UnitIsConnected(typei) then
			if UnitIsDead(typei) then
				self.NumCorpses = self.NumCorpses + 1
				if not self:UnitDoesntNeedRes(UnitName(typei)) then
					tinsert(self.ResTable, {typei, self:GetResModifier(typei)})
				else
					self.NumCorpsesRessed = self.NumCorpsesRessed + 1
				end
			else
				local name = UnitName(typei)
				if self.RessedTable[name] then
					self.RessedTable[name] = nil
				end
				if self.BlackListTable[name] then
					self.BlackListTable[name] = nil
				end
				-- TODO: Maybe clear RessingTable too
			end
		end
	end
end

function SmartRes:GetResModifier(unit)
	local name = UnitName(unit)
	local m = 0
	local _,englishClass = UnitClass(unit)
	if englishClass == "PRIEST" then m = 0
	elseif englishClass == "SHAMAN" then m = 5
	elseif englishClass == "PALADIN" then m = 5
	elseif englishClass == "DRUID" then m = 5
	elseif englishClass == "MAGE" then m = 10
	elseif englishClass == "WARLOCK" then m = 15
	elseif englishClass == "HUNTER" then m = 15
	elseif englishClass == "WARRIOR" then m = 20
	elseif englishClass == "ROGUE" then m = 20
	elseif englishClass == "DEATHKNIGHT" then m = 20
	end
	if UnitInParty(unit) then m = m - 1 end
	-- we don't want to get stuck on someone out of LOS
	m = m + math.random(5)
	if self.BlackListTable[UnitName(unit)] then
		m = m + self.BlackListTable[UnitName(unit)]
	end
	if UnitIsAFK(unit) then
		m = m + 10
	end
	--if UnitIsUnit(unit, "target") then
	--	m = m - 25
	--end
	return m
end

function SmartRes:UnitDoesntNeedRes(name)
	if self.RessedTable[name] then
		if GetTime() - self.RessedTable[name] < 60 then -- and name ~= UnitName("target")
			--self:Debug("Skipping "..name.." - Recently Ressed")
			return true
		end
	end

	for author, player in pairs(self.RessingTable) do
		if name == player then
			--self:Debug("Skipping " .. player .. " since he is being ressed by " .. author)
			return true
		end
	end

	return false
end

function SmartRes:ChooseDeadUnitInRange()
	for key, val in ipairs(self.ResTable) do
		--self:Debug("Testing Range for: "..UnitName(val[1]))
		if IsSpellInRange(self.PlayerSpell, val[1]) == 1 then
			return val[1]
		end
	end
	return nil
end

-----------------------------------------------------------
-- Helper functions
-----------------------------------------------------------
function SmartRes:CollidingRes(player, target)
	for p, t in pairs(self.RessingTable) do
		if UnitIsUnit(target, t) and not UnitIsUnit(player, p) then
			return true
		end
	end
	for p, t in pairs(self.PossibleRessingTable) do
		if UnitIsUnit(target, t) and not UnitIsUnit(player, p) then
			return true
		end
	end
	return false
end

-----------------------------------------------------------
-- UI functions
-----------------------------------------------------------
function SmartRes:OutputTextNotifications(target)
	local profile = self.db.profile
	if profile.notify then
		local targetcolored = self:ClassTextColor(target, select(2, UnitClass(target))) -- TODO error her on direct res on mage
		self:Print(format(profile.resmessage, targetcolored))
	end
	if profile.whisper then
		SendChatMessage(string.format(profile.resmessage, ""), "WHISPER", nil, target)
	end
	if profile.chattype then
		SendChatMessage(string.format(profile.resmessage, target), profile.chattype)
	end
	if profile.customchannel then
		local id, channelname = GetChannelName(profile.customchannel)
		if (id == profile.customchannel and channelname == profile.customchannelname) then
			SendChatMessage(string.format(profile.resmessage, target), "CHANNEL", nil, profile.customchannel)
		else
			self:Print("Channel [" .. profile.customchannel .. ". " .. profile.customchannelname .. "] not found")
		end
	end
end

function SmartRes:CreateBar(player, target, time, collision, test)
	local profile = self.db.profile
	if not self.Combat and profile.enable then
		local text, texture = self:GetResTextAndTexture(player, target, test)

		self:RegisterCandyBar(player, time, text, texture, profile.barcolor)
		self:SetCandyBarTexture(player, SM:Fetch('statusbar', profile.bartex))
		self:SetCandyBarBackgroundColor(player, profile.bgcolor, 0.5)
		self:SetCandyBarTextColor(player, profile.textcolor)
		self:SetCandyBarFontSize(player, profile.textsize)
		self:SetCandyBarScale(player, profile.barscale)
		self:SetCandyBarWidth(player, profile.barwidth)
		self:SetCandyBarHeight(player, profile.barheight)
		self:SetCandyBarReversed(player, profile.reverse)
		self:SetCandyBarFade(player, 0.5, true)
		self:RegisterCandyBarWithGroup(player, "SmartRes")

		if collision then
			self:SetCandyBarColor(player, profile.collisionbarcolor, 1)
		else
			self:SetCandyBarColor(player, profile.barcolor, 1)
		end
			

		self:StartCandyBar(player, true)
	end
end

function SmartRes:GetResTextAndTexture(player, target, test)
	local playercolored
	local targetcolored
	local barclass
	if not test then
		local playerclass = select(2, UnitClass(player))
		playercolored = self:ClassTextColor(player, playerclass)
		--playercolored = player
		local targetclass = select(2, UnitClass(target))
		targetcolored = self:ClassTextColor(target, targetclass)
		--targetcolored = target
		barclass = playerclass
	else
		playercolored = player
		targetcolored = target
		barclass = self.PlayerClass
	end

	local text
	if self.db.profile.shorttext then
		text = L["%s: %s"]
	else
		text = L["%s is resurrecting %s"]
	end
	text = format(text, playercolored, targetcolored)

	local texture = self.ResSpellIcons[barclass]

	return text, texture
end

function SmartRes:ClassTextColor(text, class)
	if self.db.profile.classcolors and class then
		return format("|cff%s%s|r", P:GetHex(class), text)
	else
		return text
	end
end
