----------------------------------------------------------------------------------------------------
--[[           NUMPADBAR
----------------------------------------------------------------------------------------------------

ABOUT

title:            NumPadBar
category:         Action Bars
notes:            Displays a numpad action bar. You can change its position, shape, or size.
original author:  Shaktar of Stonemaul
revival author:   Incindarella of Drak'Tharon
date:             Sept 15th, 2010
version:          4.00

CHANGE LOG

2.00  first version, numbered 2.0 to signify compatibility with WoW 2.0
2.01  
   a) code to check for dragging offscreen fixed, now can drag anwhere on screen at any resolution
   b) default keyboard now does not include enter & cursor keys, warnings about key bindings added
   c) warriors & druids no longer give up 2 action bars, just 4 rightmost slots in bottom bar #2
   d) more clear for first time users that the numpad can be left-clicked to drag and right-clicked for the menu
   e) menus simplified and clarified
   note) all users will have to rearrange their actions in the numpad once after upgrading
2.10  
   a) added stance switching for druids, priests, rogues, warriors (classes that have stance bars)
      1) druid stances: bear, cat, moonkin, treeform
      2) priest stances: shadowform
      3) rogue stances: stealth
      4) warrior stances: battle, defensive, berserker
   b) added preference to turn off stance switching for those classes
   c) added enter key binding preference and cursor key binding preference
   d) no longer flashes the background for mouseover events
   e) dragging the numpad or using the menu cause the background to display momentarily
   f) clicking on the background toggles between displaying and hiding the background
   g) fixed bug where action bars would remain locked after combat
   note) all users will have to rearrange their actions in the numpad once after upgrading
2.20  
   a) added a scrolling tooltip to display help/instructions rather than a series of dialogs
   b) major revisement to instructions, especially action bars/slots and stance switching
   c) numpad actions for druids no longer duplicate stance bar 1 when not stance switching
   d) added stance switching for moonkin and treeform (bug fix)
   e) correct stance set now shown after displaying or hiding cursor or enter keys (bug fix)
   f) fixed red dot bug in normal action bars (bug fix)
   g) added slash command /numpad which can be used instead of /numpadbar
   h) added warning to instructions regarding usage of bottom action bar #2 to save numpad actions in
   i) split cursor keys into 2 parts, navigation keys & arrow keys, for more display options
   j) fixed wow 2.2 bug: must now call update actions explicitly after buttons are created
   k) fixed wow 2.2 bug: using getleft/top instead of getpoint due to relativepoint changes
2.22  changed key label placement so that red dot range indicators can display
2.42  
   a) fixed key labels so they look like normal hotkeys again
   b) fixed slash command "/numpad help" so that it works again
   c) fixed stance switching for rogues
   d) fixed saving the position of the numpad when size is tiny
   e) fixed empty buttons so they do not flash red or outline in yellow after stance changes
   f) changed "hide" to "disable" in the menu for a better description of what that option does
   g) updated the instructions to better describe disabling (hiding) the numpad
   h) removed redundant help chat when /numpad is entered to re-enable the numpad
   i) removed the ability to make changes in combat which caused taint
   j) added compatibility with 2.4.2 changes to showgrid (to show empty buttons)
   k) renamed medium/small/tiny to large/medium/small in size menu
   l) now saves the (lower) action bar page number when using stance switching
   m) now lets moonkins/treeforms use arrow keys by using 4 slots in Right Bar 2
   n) made the scrollable tooltip function a library so other mods can use it
3.02   
   a) updated to work with echoes of doom patch
3.03
   a) fixed stances and reformatted code
3.03.1
   a) added death knight support.
3.10
   a) updated toc for 3.1
3.10.1
   a) updated rogues to not conflict with new shadow dance stance slots
3.13
   a) added ability to hide the ui while keeping it active.
3.2
   a) updated toc for 3.2
3.3
   a) updated toc for 3.3
3.33
   a) added logic to detect spec changes, specifically for druids (moonkin/tree)
   b) split apart moonkin and tree form action bar mappings
   c) changed moonkin and tree form bar mappings to more ideal action bar slots
3.33a
   a) correction to druid spec swapping... correctly rebinds keys when switching specs
3.33b
   a) correction to druid tree form stance switching
4.00
   a) updated version information
   b) removed references to 'this' (which should have been done when 2.0 came out)
   c) removed references to GetGlobal

]] -- edited with UltraEdit - Using spaces instead of tabs (and tabs set to 3)

----------------------------------------------------------------------------------------------------
-- CONSTANTS
----------------------------------------------------------------------------------------------------
NPB_VESRION = '4.00'

NPB_BACKDROP = {bgFile = 'Interface/Tooltips/UI-Tooltip-Background'}

NPB_BUTTON_SIZE = 42
NPB_DRAG_HANDLE = 8
NPB_DRAG_WIDTH = NPB_BUTTON_SIZE * 4 - 8
NPB_DRAG_HEIGHT = NPB_BUTTON_SIZE * 5 + 8

NPB_SIZES = {
   [1] = .8,
   [.8] = .67,
   [.67] = 1,
   }

NPB_MENU = {
   windows = 'Keyboard is Windows',
   office = 'Keyboard is Microsoft Office',
   multimedia = 'Keyboard is Natural Multimedia',
   elite = 'Keyboard is Natural Elite',
   macintosh = 'Keyboard is Macintosh',
   large = 'Size is Large',
   medium = 'Size is Medium',
   small = 'Size is Small',
   enter = 'Display Enter Key',
   nav = 'Display Navigation Keys',
   arrow = 'Display Arrow Keys',
   labels = 'Display Labels',
   stancebars = 'Use Stance Switching',
   disable = 'Disable Numpad',
   hide = 'Hide UI',
   help = 'Help'
}

----------------------------------------------------------------------------------------------------
-- DEFAULTS
----------------------------------------------------------------------------------------------------

-- the shape is determined by the brand of keyboard
NPB_DEFAULT_SHAPE = 'Windows' -- should be the most common, defaults to no enter/cursor keys

-- the enter key is initially not used since it would unbind open chat
NPB_DEFAULT_ENTER = ''

-- the cursor keys are initially not used since they would unbind movement keys
NPB_DEFAULT_CURSOR = ''

-- the arrow keys are initially not used since they would unbind movement keys
NPB_DEFAULT_ARROW = ''

-- the navigation keys are initially not used since they would unbind movement keys
NPB_DEFAULT_NAV = ''

-- size is 1 for normal action button size, or smaller for smaller buttons
-- for example, .8 would be 80% of normal size
NPB_DEFAULT_SIZE = 1       -- initially normal button size

-- save the position of the numpad frame
NPB_DEFAULT_LEFT = 750
NPB_DEFAULT_TOP = 350

-- labels are shortened versions of key bindings, example: NL for NumLock
NPB_DEFAULT_LABELS = true     -- display labels

-- this is set to true after the mistakenly filled actions are cleared before creating numpads
NPB_DEFAULT_CLEARED = false

-- this is set to true after the user left-clicks or right-clicks on the numpad background
-- while this is false, messages tell the user that they can left-click or right-click on the background
NPB_DEFAULT_CLICKED = false

----------------------------------------------------------------------------------------------------
-- SAVED VARIABLES
----------------------------------------------------------------------------------------------------

NumPadBar = { }                     -- defaults will be filled in by CheckSavedValues()
NumPadBar.cleared = NPB_DEFAULT_CLEARED   -- except for this default value
NumPadBar.leftclicked = NPB_DEFAULT_CLICKED  -- and this one
NumPadBar.rightclicked = NPB_DEFAULT_CLICKED -- and this one

----------------------------------------------------------------------------------------------------
-- GLOBAL VARIABLES
----------------------------------------------------------------------------------------------------

NPB_numpads = { }       -- table of numpads (frames with buttons)
NPB_lockbars = nil            -- the original value for LOCK_ACTIONBAR
NPB_clickme = false           -- so that clickme message displayed only once per session
NPB_class = ''             -- the character's class such as warrior
NPB_numpad = nil
NPB_login = '0'

----------------------------------------------------------------------------------------------------
-- MESSAGES
----------------------------------------------------------------------------------------------------

NPB_PASTEL = '|cffffff80'     -- pastel yellow
NPB_BRIGHT = '|cffffff00'     -- bright yellow
NPB_PLAIN = '|r'        -- reset color

NPB_TITLE = NPB_PASTEL .. 'NumPadBar: ' .. NPB_PLAIN

NPB_DESCRIPTION = NPB_BRIGHT .. 'NumPadBar ' .. NPB_VESRION .. ': ' .. NPB_PASTEL .. 'Displays a numpad action bar. You can change its position, shape, or size. Move the numpad by dragging the background. Display the drop down menu by right-clicking the background. For instructions, select help from the menu or type /numpad help.'

NPB_DISABLED = NPB_TITLE .. 'Disabled. To enable it, type /numpad'
NPB_HIDDEN = NPB_TITLE .. 'Bar hidden. To show it type /numpad show'
NPB_RESET = NPB_TITLE .. 'All saved NumPadBar values are now reset to default.'

NPB_CLICK_ERROR = 'Left-Click to drag numpad, Right-Click to display menu'
NPB_ENTER_UNBOUND = 'Open Chat (enter key) now unbound!'
NPB_NAV_UNBOUND = 'Chat scroll & camera angle (navigation) keys now unbound!'
NPB_ARROW_UNBOUND = 'Movement (arrow) keys now unbound!'

NPB_USING_STANCEBARS = NPB_TITLE .. 'The numeric keys and the decimal point key on the numpad will now switch to the appropriate stance bar on stance changes; keys 0-9 and [.] on the numpad will match keys 0-9 and [-] on the stance bar. Non-numeric keys on the numpad will always have the same action no matter what stance.'
NPB_NOT_USING_STANCEBARS = NPB_TITLE .. 'Keys on the numpad will no longer switch on stance changes.'
NPB_STANCELESS_CLASS = NPB_TITLE .. 'Only druids, priests, rogues, and warriors have stance bars. Your class does not use stance bars.'

NPB_ARROW_TOPSLOTS = NPB_TITLE .. 'Top 4 slots of Right Bar 2 now used by numpad'
NPB_ARROW_RIGHTSLOTS = NPB_TITLE .. 'Rightmost 4 slots of ActionBar 2 now used by numpad'
NPB_MOONKIN_BAR = NPB_TITLE .. 'Moonkin and treeform druids lack enough spare slots to save numpad actions in. The entire ActionBar 2 will be use by NumPadBar to save actions in.'

----------------------------------------------------------------------------------------------------
-- ACTION BAR SLOTS
----------------------------------------------------------------------------------------------------

-- 1 to 12     default action bar 1    standard ActionBar 1 (not used by warriors)
-- 13 to 24    default action bar 2    standard ActionBar 2
-- 25 to 36    default action bar 3    Right Bar (rightmost of the 2 right bars)
-- 37 to 48    default action bar 4    Right Bar 2 (leftmost of the 2 right bars)
-- 49 to 60    default action bar 5    Bottom Right Bar
-- 61 to 72    default action bar 6    Bottom Left Bar
-- 73 to 84    warrior battle stance, druid cat form, rogue stealth, priest shadowform
-- 85 to 96    warrior defensive stance, druid tree form, rogue shadowdance
-- 97 to 108   warrior berserker stance, druid bear form
-- 109 to 120  druid moonkin form

NPB_NORMAL_SLOTS = {
   85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96,
   97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108,
   109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120
}

NPB_DRUID_SLOTS = {   -- everything but Tree/Moonkin
   85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96,
   109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
   24, 23, 22, 21     -- action bar 2 slots
}

NPB_TREE_SLOTS = {
   109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
   48, 47, 46, 45, 13, 14, 15, 16, 17, 18, 19, 20,    -- four action bar 4 slots and action bar 2 slots
   21, 22, 23, 24                                     -- remainder of action bar 2 slots
}

NPB_MOONKIN_SLOTS = {
   13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24,    -- action bar 2 slots
   36, 35, 34, 33, 32, 31, 30, 29, 28, 27, 26, 25,    -- action bar 3 slots, backwards
   37, 38, 39, 40     -- action bar 4 slots (Right Bar 2, leftmost of the 2 right bars)
}

NPB_ROGUE_SLOTS = {
   109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
   97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108,
   96, 95, 94, 93     -- Shadow dance stance bar (4 right most slots)
}

NPB_WARRIOR_SLOTS = {
   1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12,
   109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120,
   24, 23, 22, 21     -- action bar 2 slots
}

NPB_CLASS_SLOTS = {
   deathknight = NPB_NORMAL_SLOTS,
   druid = NPB_DRUID_SLOTS,
   hunter = NPB_NORMAL_SLOTS,
   mage = NPB_NORMAL_SLOTS,
   moonkin = NPB_MOONKIN_SLOTS,
   paladin = NPB_NORMAL_SLOTS,
   priest = NPB_NORMAL_SLOTS,
   rogue = NPB_ROGUE_SLOTS,
   shaman = NPB_NORMAL_SLOTS,
   tree = NPB_TREE_SLOTS,
   warlock = NPB_NORMAL_SLOTS,
   warrior = NPB_WARRIOR_SLOTS
}

----------------------------------------------------------------------------------------------------
-- STANCE NUMBERS
-- see http://www.wowwiki.com/API_GetShapeshiftForm
----------------------------------------------------------------------------------------------------

--[[
Class:   Warrior   Priest  Rogue       Druid
-------------------------------------------------------------------------------------
Stance:1 Battle    Shadow  Stealth     Bear
Stance:2 Defensive                     Aquatic
Stance:3 Berserker                     Cat
Stance:4                               Travel
Stance:5                               Moonkin/Tree/Flight*
Stance:6                               Flight
*If you have both e.g. Moonkin form and Flying form then Moonkin is 5 and Flight is 6.
]]

----------------------------------------------------------------------------------------------------
-- STANCE BARS
-- see http://www.wowwiki.com/ActionSlot
----------------------------------------------------------------------------------------------------

NPB_CLASS_STANCEBARS = {druid = 1, moonkin = 2, priest = 3, rogue = 4, warrior = 5, tree = 6}

NPB_STANCES = {
   [1] = { 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84 },
   [2] = { 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96 },
   [3] = { 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108 }
}
-- [4] = { 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120 }

----------------------------------------------------------------------------------------------------
-- KEYBOARD LAYOUTS
----------------------------------------------------------------------------------------------------

NPB_KEYBOARDS = {

Windows = {

   { row = 4, col = 1, type='Numeric', label = '1', binding = 'NumPad1' },
   { row = 4, col = 2, type='Numeric', label = '2', binding = 'NumPad2' },
   { row = 4, col = 3, type='Numeric', label = '3', binding = 'NumPad3' },
   { row = 3, col = 1, type='Numeric', label = '4', binding = 'NumPad4' },
   { row = 3, col = 2, type='Numeric', label = '5', binding = 'NumPad5' },
   { row = 3, col = 3, type='Numeric', label = '6', binding = 'NumPad6' },
   { row = 2, col = 1, type='Numeric', label = '7', binding = 'NumPad7' },
   { row = 2, col = 2, type='Numeric', label = '8', binding = 'NumPad8' },
   { row = 2, col = 3, type='Numeric', label = '9', binding = 'NumPad9' },
   { row = 5, col = 1.5, type='Numeric', label = '0', binding = 'NumPad0' },

   { row = 5, col = 3, type='Numeric', label = '.', binding = 'NumPadDecimal' },
   { row = 2.5, col = 4, type='Numeric', label = '+', binding = 'NumPadPlus' },
   { row = 1, col = 4, type='Numeric', label = '-', binding = 'NumPadMinus' },
   { row = 1, col = 3, type='Numeric', label = '*', binding = 'NumPadMultiply' },
   { row = 1, col = 2, type='Numeric', label = '/', binding = 'NumPadDivide' },

   { row = 4.5, col = 4, type='Enter', label = 'E', binding = 'Enter' },

   { row = 1, col = 1 - 3.3, type='Nav', label = 'I', binding = 'Insert' },
   { row = 1, col = 2 - 3.3, type='Nav', label = 'H', binding = 'Home' },
   { row = 1, col = 3 - 3.3, type='Nav', label = 'P^', binding = 'PageUp' },
   { row = 2, col = 1 - 3.3, type='Nav', label = 'D', binding = 'Delete' },
   { row = 2, col = 2 - 3.3, type='Nav', label = 'E', binding = 'End' },
   { row = 2, col = 3 - 3.3, type='Nav', label = 'Pv', binding = 'PageDown' },

   { row = 4, col = 2 - 3.3, type='Arrow', label = '^', binding = 'Up' },
   { row = 5, col = 1 - 3.3, type='Arrow', label = '<', binding = 'Left' },
   { row = 5, col = 2 - 3.3, type='Arrow', label = 'v', binding = 'Down' },
   { row = 5, col = 3 - 3.3, type='Arrow', label = '>', binding = 'Right' }
   
},

MicrosoftOffice = {

   { row = 4, col = 1, type='Numeric', label = '1', binding = 'NumPad1' },
   { row = 4, col = 2, type='Numeric', label = '2', binding = 'NumPad2' },
   { row = 4, col = 3, type='Numeric', label = '3', binding = 'NumPad3' },
   { row = 3, col = 1, type='Numeric', label = '4', binding = 'NumPad4' },
   { row = 3, col = 2, type='Numeric', label = '5', binding = 'NumPad5' },
   { row = 3, col = 3, type='Numeric', label = '6', binding = 'NumPad6' },
   { row = 2, col = 1, type='Numeric', label = '7', binding = 'NumPad7' },
   { row = 2, col = 2, type='Numeric', label = '8', binding = 'NumPad8' },
   { row = 2, col = 3, type='Numeric', label = '9', binding = 'NumPad9' },
   { row = 5, col = 1.5, type='Numeric', label = '0', binding = 'NumPad0' },

   { row = 5, col = 3, type='Numeric', label = '.', binding = 'NumPadDecimal' },
   { row = 2.5, col = 4, type='Numeric', label = '+', binding = 'NumPadPlus' },
   { row = 1, col = 4, type='Numeric', label = '-', binding = 'NumPadMinus' },
   { row = 1, col = 3, type='Numeric', label = '*', binding = 'NumPadMultiply' },
   { row = 1, col = 2, type='Numeric', label = '/', binding = 'NumPadDivide' },

   { row = 4.5, col = 4, type='Enter', label = 'E', binding = 'Enter' },

   { row = 1, col = 2 - 3.3, type='Nav', label = 'H', binding = 'Home' },
   { row = 1, col = 3 - 3.3, type='Nav', label = 'E', binding = 'End' },
   { row = 2, col = 3 - 3.3, type='Nav', label = 'P^', binding = 'PageUp' },
   { row = 2.5, col = 2 - 3.3, type='Nav', label = 'D', binding = 'Delete' },
   { row = 3, col = 3 - 3.3, type='Nav', label = 'Pv', binding = 'PageDown' },

   -- the following key is extra on a windows office keyboard
   { row = 1, col = 1, type='Nav', label = 'T', binding = 'Tab' },

   { row = 4.3, col = 2 - 3.3, type='Arrow', label = '^', binding = 'Up' },
   { row = 5.3, col = 1 - 3.3, type='Arrow', label = '<', binding = 'Left' },
   { row = 5.3, col = 2 - 3.3, type='Arrow', label = 'v', binding = 'Down' },
   { row = 5.3, col = 3 - 3.3, type='Arrow', label = '>', binding = 'Right' }
   
},

NaturalMultimedia = {

   { row = 4, col = 1, type='Numeric', label = '1', binding = 'NumPad1' },
   { row = 4, col = 2, type='Numeric', label = '2', binding = 'NumPad2' },
   { row = 4, col = 3, type='Numeric', label = '3', binding = 'NumPad3' },
   { row = 3, col = 1, type='Numeric', label = '4', binding = 'NumPad4' },
   { row = 3, col = 2, type='Numeric', label = '5', binding = 'NumPad5' },
   { row = 3, col = 3, type='Numeric', label = '6', binding = 'NumPad6' },
   { row = 2, col = 1, type='Numeric', label = '7', binding = 'NumPad7' },
   { row = 2, col = 2, type='Numeric', label = '8', binding = 'NumPad8' },
   { row = 2, col = 3, type='Numeric', label = '9', binding = 'NumPad9' },
   { row = 5, col = 1.5, type='Numeric', label = '0', binding = 'NumPad0' },

   { row = 5, col = 3, type='Numeric', label = '.', binding = 'NumPadDecimal' },
   { row = 2.5, col = 4, type='Numeric', label = '+', binding = 'NumPadPlus' },
   { row = 1, col = 4, type='Numeric', label = '-', binding = 'NumPadMinus' },
   { row = 1, col = 3, type='Numeric', label = '*', binding = 'NumPadMultiply' },
   { row = 1, col = 2, type='Numeric', label = '/', binding = 'NumPadDivide' },

   { row = 4.5, col = 4, type='Enter', label = 'E', binding = 'Enter' },

   { row = 1, col = 2 - 3.3, type='Nav', label = 'H', binding = 'Home' },
   { row = 1, col = 3 - 3.3, type='Nav', label = 'E', binding = 'End' },
   { row = 2, col = 3 - 3.3, type='Nav', label = 'P^', binding = 'PageUp' },
   { row = 2.5, col = 2 - 3.3, type='Nav', label = 'D', binding = 'Delete' },
   { row = 3, col = 3 - 3.3, type='Nav', label = 'Pv', binding = 'PageDown' },

   { row = 4.3, col = 2 - 3.3, type='Arrow', label = '^', binding = 'Up' },
   { row = 5.3, col = 1 - 3.3, type='Arrow', label = '<', binding = 'Left' },
   { row = 5.3, col = 2 - 3.3, type='Arrow', label = 'v', binding = 'Down' },
   { row = 5.3, col = 3 - 3.3, type='Arrow', label = '>', binding = 'Right' }
   
},

NaturalElite = {

   { row = 4, col = 1, type='Numeric', label = '1', binding = 'NumPad1' },
   { row = 4, col = 2, type='Numeric', label = '2', binding = 'NumPad2' },
   { row = 4, col = 3, type='Numeric', label = '3', binding = 'NumPad3' },
   { row = 3, col = 1, type='Numeric', label = '4', binding = 'NumPad4' },
   { row = 3, col = 2, type='Numeric', label = '5', binding = 'NumPad5' },
   { row = 3, col = 3, type='Numeric', label = '6', binding = 'NumPad6' },
   { row = 2, col = 1, type='Numeric', label = '7', binding = 'NumPad7' },
   { row = 2, col = 2, type='Numeric', label = '8', binding = 'NumPad8' },
   { row = 2, col = 3, type='Numeric', label = '9', binding = 'NumPad9' },
   { row = 5, col = 1.5, type='Numeric', label = '0', binding = 'NumPad0' },

   { row = 5, col = 3, type='Numeric', label = '.', binding = 'NumPadDecimal' },
   { row = 2.5, col = 4, type='Numeric', label = '+', binding = 'NumPadPlus' },
   { row = 1, col = 4, type='Numeric', label = '-', binding = 'NumPadMinus' },
   { row = 1, col = 3, type='Numeric', label = '*', binding = 'NumPadMultiply' },
   { row = 1, col = 2, type='Numeric', label = '/', binding = 'NumPadDivide' },

   { row = 4.5, col = 4, type='Enter', label = 'E', binding = 'Enter' },

   { row = 1, col = 1 - 2.6, type='Nav', label = 'H', binding = 'Home' },
   { row = 1, col = 2 - 2.6, type='Nav', label = 'P^', binding = 'PageUp' },
   { row = 2, col = 1 - 2.6, type='Nav', label = 'E', binding = 'End' },
   { row = 2, col = 2 - 2.6, type='Nav', label = 'Pv', binding = 'PageDown' },
   { row = 3, col = 1 - 2.6, type='Nav', label = 'D', binding = 'Delete' },
   { row = 3, col = 2 - 2.6, type='Nav', label = 'I', binding = 'Insert' },

   { row = 4.3, col = 2 - 3.1, type='Arrow', label = '^', binding = 'Up' },
   { row = 4.8, col = 1 - 3.1, type='Arrow', label = '<', binding = 'Left' },
   { row = 5.3, col = 2 - 3.1, type='Arrow', label = 'v', binding = 'Down' },
   { row = 4.8, col = 3 - 3.1, type='Arrow', label = '>', binding = 'Right' }

},
   
Macintosh = {

   { row = 4, col = 1, type='Numeric', label = '1', binding = 'NumPad1' },
   { row = 4, col = 2, type='Numeric', label = '2', binding = 'NumPad2' },
   { row = 4, col = 3, type='Numeric', label = '3', binding = 'NumPad3' },
   { row = 3, col = 1, type='Numeric', label = '4', binding = 'NumPad4' },
   { row = 3, col = 2, type='Numeric', label = '5', binding = 'NumPad5' },
   { row = 3, col = 3, type='Numeric', label = '6', binding = 'NumPad6' },
   { row = 2, col = 1, type='Numeric', label = '7', binding = 'NumPad7' },
   { row = 2, col = 2, type='Numeric', label = '8', binding = 'NumPad8' },
   { row = 2, col = 3, type='Numeric', label = '9', binding = 'NumPad9' },
   { row = 5, col = 1.5, type='Numeric', label = '0', binding = 'NumPad0' },

   { row = 5, col = 3, type='Numeric', label = '.', binding = 'NumPadDecimal' },
   { row = 3, col = 4, type='Numeric', label = '+', binding = 'NumPadPlus' },
   { row = 2, col = 4, type='Numeric', label = '-', binding = 'NumPadMinus' },
   { row = 1, col = 4, type='Numeric', label = '*', binding = 'NumPadMultiply' },
   { row = 1, col = 3, type='Numeric', label = '/', binding = 'NumPadDivide' },
   
   -- the following 2 keys are extra on a macintosh
   -- + and - key on mac take up same room as + key on pc
   { row = 1, col = 2, type='Numeric', label = '=', binding = 'NumPadEquals' },
   { row = 1, col = 1, type='Numeric', label = 'C', binding = 'NumLock' },   -- Clear

   { row = 4.5, col = 4, type='Enter', label = 'E', binding = 'Enter' },
   
   { row = 1, col = 1 - 3.3, type='Nav', label = 'H?', binding = 'Insert' },   -- Help
   { row = 1, col = 2 - 3.3, type='Nav', label = 'H', binding = 'Home' },
   { row = 1, col = 3 - 3.3, type='Nav', label = 'P^', binding = 'PageUp' },
   { row = 2, col = 1 - 3.3, type='Nav', label = 'D', binding = 'Delete' },
   { row = 2, col = 2 - 3.3, type='Nav', label = 'E', binding = 'End' },
   { row = 2, col = 3 - 3.3, type='Nav', label = 'Pv', binding = 'PageDown' },

   { row = 4, col = 2 - 3.3, type='Arrow', label = '^', binding = 'Up' },
   { row = 5, col = 1 - 3.3, type='Arrow', label = '<', binding = 'Left' },
   { row = 5, col = 2 - 3.3, type='Arrow', label = 'v', binding = 'Down' },
   { row = 5, col = 3 - 3.3, type='Arrow', label = '>', binding = 'Right' }

}

}

----------------------------------------------------------------------------------------------------
-- INSTRUCTIONS
----------------------------------------------------------------------------------------------------

NPB_INSTRUCTIONS =

[[
<big>NumPadBar</big> <i>by Shaktar of Stonemaul and Incindarella of DrakTharon</i>

<b>Features</b>

* Offers a choice of five popular keyboards
* Offers a choice of three action bar sizes
* Switches actions on form/stance changes if desired
* Hides the labels on the buttons if desired
* Locks all action bars during combat
* Prevents shift-dragging actions out of numpad buttons during combat
* Prevents dragging numpad off screen
* Saves settings individually for each character
* Displays detailed instructions in a scrollable tooltip
* Disables the numpad by removing it and its keybindings
* Compatible with OmniCC, a ui mod that displays cooldowns on buttons

<b>Warning</b>

Due to the lack of spare action slots for druids and warriors, ActionBar 2 (bar page 2 of the standard action bar) is used to store numpad actions in. Additionally, for moonkin and treeform druids, the top 4 slots of Right Bar 2 (leftmost of the right action bars) are used.

<b>Description</b>

NumPadBar is an action bar in the shape of a numpad. It works like a built-in action bar such as the Bottom Right Bar, for example. You can drag actions and macros into and out of NumPadBar, just like other action bars.

The first major difference between NumPadBar and built-in action bars is that it is in the shape of a numpad. Also, keybindings for numpad keys are already set up for the numpad bar. These keybindings are temporary and are only used while the numpad is displayed.

The second major difference is that NumPadBar can be used as a normal action bar or as a stance bar. When using it as a normal action bar, the same actions are used no matter what stance you are in. When using it as a stance bar, the actions will change depending on your stance; they will be the same actions that are in the built-in stance bar.

You can move NumPadBar anywhere on screen. You can change its size. You can choose from a selection of popular keyboards; this affects the layout of the numpad keys. You can hide the labels on the keys. You can select whether to display the Enter key (and override Chat binding), and whether to display the cursor keys (and override movement and other bindings). You can disable it which removes the numpad bar and its keybindings.

In the heat of combat, it is all too easy to accidentally drag actions out of action bars. Therefore, NumPadBar, as well as the built-in action bars, are locked during combat. This prevents all changes to NumPadBar during combat.

You can display instructions in a scrollable tooltip by right-clicking the invisible background of NumPadBar near the numeric keys and selecting Help. To scroll the instructions, use your mouse wheel.

<b>Instructions</b>

You can move the numpad anywhere on the screen. It can overlap built-in action bars, although its buttons would be on a lower plane than buttons on the built-in bars. To move the numpad, click and drag the invisible background of the numpad near the numeric keys. The background will temporarily turn gray and you can drag the numpad to a different place on your screen.

You can set various preferences by using the drop down menu. To display the drop down menu, right-click the invisible background of the numpad near the numeric keys. The background will temporarily turn gray and the drop down menu will appear. You can set various options with this menu such as changing the shape or size of the numpad, or hiding the numpad or labels. Select the desired preference from this menu.

You can drag actions and macros into and out of the numpad buttons. Action bars must be unlocked and you must be out of combat.

Buttons are not pre-filled with any actions. If you see any actions already in the numpad buttons before you have dragged any actions into the buttons, it will be because you previously installed a different action bar mod and that mod has used the same slots that are used by the numpad. If other action bar mods are currently installed, you will have problems with other action bars using the same slots as the numpad.

To display cooldowns on the buttons, you can use a cooldown mod such as OmniCC.

<b>Action Bars & Slots</b>

There are only 120 slots to save actions in. Most of these slots are taken up by the built-in action bars as well as the stance bars that some classes have. There are a very limited number of spare slots to use for the numpad bar, but there are enough for most classes. However, warriors and druids have more stance bars than other classes and therefore less slots to spare.  Moonkin and treeform druids have even less spare slots.

For warriors and feral druids, there is a shortage of 4 slots. For moonkin and treeform druids, there is a shortage of 16 slots. These slots must come from built-in action bars. ActionBar 2 (bar page 2 of the standard action bar) is used to save numpad actions in since this bar is normally not displayed. The top 4 slots of Right Bar 2 (the leftmost of the right action bars) are also used since this bar is the least popular of the built-in bars.

For moonkin and treeform druids: The entire ActionBar 2 is used to save numpad actions in. You should not use this bar. Additionally, the top 4 slots in Right Bar 2 are used to save arrow actions in.

For warriors and feral druids using arrow keys: The rightmost 4 slots in ActionBar 2 are used to save arrow actions in. If you place actions in those slots, you will overwrite actions in the numpad.

For all other classes: There are enough unused slots to save numpad actions in. Slots from existing built-in action bars are not needed.

In summary, slots in the following action bars are used to save numpad actions in:
* For moonkin & treeform druids, the entire ActionBar 2 & the top 4 slots in Right Bar 2
* For warriors & feral druids using arrow keys, the rightmost 4 slots of ActionBar 2
* For warriors & feral druids not using arrow keys, no slots are used
* For all other classes, no slots are used

<b>Stance Switching Enabled</b>

Stance switching is optional and is supported for druids, priests, rogues, and warriors, but only for stances/forms that have built-in stance bars. Actions on the numpad will match most actions on the built-in stance bar. This is for people that prefer to use the numpad rather than the normal numeric keys when using the stance bar.

When using NumPadBar as a stance bar, the numeric keys on the numpad will have the same actions as the built-in stance bar (bar #1). Specifically, keys 0-9 and decimal point [.] on the numpad will be the same as keys 0-9 and minus key [-] on the normal keyboard.

To avoid seeing two essentially identical versions of your stance bar (normal numeric keys and numpad), you can scroll to a different action bar. This will let you use your numpad as your stance bar while using the normal numeric keys for other actions. You can scroll to bar #2 as long as you keep in mind that some slots in this bar are used by the numpad as mentioned above. Or you can scroll to bar #3 or higher and avoid this problem.

If you have previously chosen to show all of the built-in action bars, you will not have a bar #3. If you wish to use a bar higher than bar #2, you will have to hide at least one of the built-in action bars.

The following stance bars are supported:
* Druids have stance bars for bear, cat, moonkin, and treeform.
* Priests have a stance bar for shadowform.
* Rogues have a stance bar for stealth.
* Warriors have stance bars for battle, defensive, and berserker stances.

The following happens if you change stance and have chosen stance switching:
* Character goes into a different stance/form
* Keys 0-9 on numpad change to match keys 0-9 on built-in stance bar
* Decimal point on numpad changes to match minus key on built-in stance bar
* Operator keys (+-*/), enter key, cursor keys (insert, delete, etc) remain the same

<b>Stance Switching Disabled</b>

All classes can choose to disable stance switching. The numpad will be a normal action bar, not a stance bar. Actions on the numpad will remain the same no matter what stance or form is selected.

The following happens if you disable stance switching:
* Slots in ActionBar 2 and Right Bar 2 will be used to save numpad actions in
* The actions in the numpad remain the same no matter what stance or form you switch to

<b>NumLock Key</b>

If your PC does not start up with your keyboard in NumLock mode, you can enable it in your BIOS. Restart your PC and (depending on your PC) either Del, F1, F2, F10, Esc, Ctl-S, or your PCs BIOS setup key before windows begins to load. Then set the NumLock option in BIOS to on and exit the BIOS.

Pressing the NumLock key while in NumLock mode changes the numpad to a cursor keypad. For example, shift-8 remaps to up-arrow and shift-1 remaps to End. If you accidentally press NumLock, then pressing numpad keys will no longer perform the actions you expect but will have completely different actions. You will have to press the NumLock key again to change it back to normal.

To avoid this problem permanently, you can use a keyboard remapping app (exe) such as KeyTweak to either disable or remap the NumLock button on your keyboard. At least disable this key so that this problem will never happen. Or better yet, remap it to the Escape key, so that you can use it as a Clear key when using calculator apps. You can use NumPadBar without remapping the NumLock key; just avoid pressing it.

Holding the shift key down while pressing numpad keys has the same effect as pressing the NumLock key and then pressing numpad keys. Unexpected actions will be performed. This problem cannot be avoided using a keyboard remapper. It can only be avoided by never holding the shift key when pressing numpad keys.

The NumLock key is used by WoW as the default key binding for Toggle Autorun. If you disable this key to avoid the problems mentioned above but still want a key for Toggle Autorun, you will have to set a new key binding for it.

<b>Key Bindings</b>

All of the numeric/operator key bindings are used for the numpad. This overrides numpad-plus and numpad-minus for minimap zoom.

The Enter key can optionally be displayed in the numpad. If displayed, the Enter key binding is used for the numpad, overriding Open Chat. (The regular Enter key and the numpad Enter key are considered the same by WoW.) You will have to set a new key binding for the Open Chat function. The apostrophe key makes for a good key binding for Open Chat.

The Cursor keys can optionally be displayed. If displayed, the cursor and arrow key bindings are used for the numpad, overriding Previous Action Bar, Next Action Bar, movement (arrow) keys, chat scroll functions, and camera angle functions.

Both the numpad key bindings and the shifted numpad key bindings are used. So, for example, pressing numpad-plus results in the same action as pressing shift-numpad-plus. This helps prevent unexpected behavior especially during combat. 

If you disable the numpad or de-install NumPadBar, all of the normal WoW numpad key bindings will be restored to their original bindings.

You can still use the alt key for self-casting.

<b>Mouse Bindings</b>

Previous Action Bar and Next Action Bar are bound by default to shift-mousewheelup and shift-mousewheeldown. When moving your hand from the numpad to the mouse, it is too easy to hit those bindings and accidentally scroll the bottom action bar.

To avoid this problem, you should unbind Previous Action Bar and Next Action Bar using the built-in WoW key bindings interface panel. You can rebind them to keys other than numpad keys.

<b>Summary</b>

(1) Do not use the NumLock key (or remap it)
(2) Do not hold the shift key down when using the numpad
(3) If you choose to display the enter key, rebind Open Chat
(4) Unbind the default mouse bindings for Previous Action Bar and Next Action Bar
(5) Keep in mind ActionBar 2 and Right Bar 2 are used to save numpad actions in
]]

----------------------------------------------------------------------------------------------------
-- WoW FUNCTIONS (for hooksecurefunc)
----------------------------------------------------------------------------------------------------

----------------------------------------------------------------------------------------------------
function NPB_DisplayHotkey(button)
   local pad = button.npb_pad
   local label = button.npb_label
   if not button.npb_pad then return end
   local hotkey = _G[button:GetName()..'HotKey']
   if label == '' then
      hotkey:SetText(RANGE_INDICATOR)
      hotkey:SetPoint('TOPLEFT', button, 'TOPLEFT', 1, -2)
      hotkey:Hide()
   else
      hotkey:SetText(label)
      hotkey:SetPoint('TOPLEFT', button, 'TOPLEFT', -2, -2)
      hotkey:Show()
   end
end

----------------------------------------------------------------------------------------------------
-- for hooksecurefunc('ActionButton_UpdateHotkeys', NPB_UpdateHotkeys)
function NPB_UpdateHotkeys(self, actionButtonType)
   NPB_DisplayHotkey(self)
end

----------------------------------------------------------------------------------------------------
-- NUMPADBAR FUNCTIONS
----------------------------------------------------------------------------------------------------

----------------------------------------------------------------------------------------------------
-- returns system time in seconds
function NPB_time()
   if GetTime then return GetTime() end
   if os and os.time then return os.time() end
   return 0
end

----------------------------------------------------------------------------------------------------
-- displays message in the chat frame
function NPB_Chat(message)
   if not message or not DEFAULT_CHAT_FRAME then return end
   DEFAULT_CHAT_FRAME:AddMessage(message)
end

----------------------------------------------------------------------------------------------------
function NPB_DoSlash(cmd)
   cmd = string.lower(cmd)
   
   if cmd == 'help' then
      NPB_Help()
      return end
      
   if cmd == 'show' then
      NPB_Show()
      return end
      
   if cmd == 'hide' then
      NPB_Hide()
      return end
   
   -- displays the key bound to the Jump action in a string that can be used in code
   -- this may not be the same as the string displayed to the user in the keybindings dialog
   -- this slash command is only useful for development, to get keybinding strings for various keys
   if cmd == 'jump' then
      NPB_Chat(GetBindingKey('Jump'))
      return end
   
   -- for development purposes, lists all of the used slots
   if cmd == 'slots' then
      NPB_ListSlots()
      return end
   
   -- for development purposes, resets all values to default
   if cmd == 'reset' then
      NPB_HideNumpad()
      NumPadBar = { }
      NPB_clickme = false
      NPB_CheckValues()
      NPB_ShowNumpad()
      NPB_Chat(NPB_RESET)
      return end
   
   local numpad = NPB_GetNumpad()
   if not NumPadBar.leftclicked or not NumPadBar.rightclicked then
      NPB_SetBackdrop(numpad)
      NPB_Chat(NPB_DESCRIPTION)
   end
   NPB_ShowNumpad()
end

----------------------------------------------------------------------------------------------------
function NPB_ListSlots()
   for slot = 1, 120 do
      local texture = GetActionTexture(slot)
      if texture then
         NPB_Chat(slot .. ': ' .. texture) end
   end
end

----------------------------------------------------------------------------------------------------
function NPB_DoLoad(self)
   self:RegisterEvent('PLAYER_LOGIN')
   self:RegisterEvent('PLAYER_LOGOUT')
   self:RegisterEvent('CHAT_MSG_CHANNEL_NOTICE')
   self:RegisterEvent('PLAYER_REGEN_DISABLED')
   self:RegisterEvent('PLAYER_REGEN_ENABLED')
   self:RegisterEvent('ACTIVE_TALENT_GROUP_CHANGED')
   SlashCmdList['NPB'] = NPB_DoSlash
   SLASH_NPB1 = '/numpadbar'
   SLASH_NPB2 = '/numpad'
   hooksecurefunc('ActionButton_UpdateHotkeys', NPB_UpdateHotkeys)
end

----------------------------------------------------------------------------------------------------
function NPB_DoEvent(self, event)
   if event == 'PLAYER_LOGIN' then
      NPB_Initialize() 
      NPB_login = '1'
   end
   
   if event == 'PLAYER_LOGOUT' then
      NPB_LogOut() end
   
   if event == 'CHAT_MSG_CHANNEL_NOTICE' then
      NPB_Notify()
      self:UnregisterEvent('CHAT_MSG_CHANNEL_NOTICE')
      end
      
   if NPB_login == '1' then
      if event == 'ACTIVE_TALENT_GROUP_CHANGED' then
         NPB_HideNumpad()
         NPB_GetClass()
         NPB_SetState()
         NPB_ShowNumpad()
      end
   end
   
   -- stop event handling if numpad is disabled
   if NumPadBar.disabled then return end
   
   -- in combat, so lock action bars
   if event == 'PLAYER_REGEN_DISABLED' then
      NPB_LockBars() end
   
   -- out of combat, so unlock bars
   if event == 'PLAYER_REGEN_ENABLED' then
      NPB_UnlockBars() end
end

----------------------------------------------------------------------------------------------------
function NPB_Initialize()
   NPB_GetClass()
   NPB_CheckValues()
   NPB_ClearSlots()
   if NumPadBar.stancebars and not NumPadBar.disabled then
      ChangeActionBarPage(NumPadBar.barpage) end
   if not NumPadBar.disabled then
      NPB_ShowNumpad() end
end

----------------------------------------------------------------------------------------------------
function NPB_Notify()
   if NumPadBar.disabled then
      NPB_Chat(NPB_DISABLED) end
end

----------------------------------------------------------------------------------------------------
function NPB_LogOut()
   NumPadBar.barpage = 1
   if NumPadBar.stancebars and not NumPadBar.disabled then
      NumPadBar.barpage = GetActionBarPage() end
end

----------------------------------------------------------------------------------------------------
function NPB_LockBars()
   if LOCK_ACTIONBAR == '1' then return end
   local numpad = NPB_GetNumpad()
   NPB_ClearBackdrop(numpad)     -- remove gray bg, don't restore it when out of combat
   LOCK_ACTIONBAR = '1'
   NumPadBar.unlock = true
end

----------------------------------------------------------------------------------------------------
function NPB_UnlockBars()
   if not NumPadBar.unlock then return end
   LOCK_ACTIONBAR = '0'
   NumPadBar.unlock = false
end

----------------------------------------------------------------------------------------------------
-- sets a global called NPB_class to the character's class
-- sets it to moonkin for moonkin druids and treeform druids
-- moonkins/trees use an extra default stancebar and therefore have less spare action slots
function NPB_GetClass()
   local localizedClass, englishClass = UnitClass('player')
   NPB_class = string.lower(englishClass)
   for fx = 1, GetNumShapeshiftForms() do
      local icon, name, isActive = GetShapeshiftFormInfo(fx)
      if name == 'Moonkin Form' then
         NPB_class = 'moonkin' end
      if name == 'Tree of Life' then
         NPB_class = 'tree' end
   end
end

----------------------------------------------------------------------------------------------------
-- a couple of slots are filled with actions for certain classes, a blizzard bug
-- for warriors, slots 1 and 109 are mistakenly prefilled
-- for druids, slots 85 and 109 are mistakenly prefilled
-- those actions are cleared since the slots are used by numpadbar
-- this only happens once per character, before the numpads are created
function NPB_ClearSlots()
   if NumPadBar.cleared then return end
   
   if NPB_class == 'warrior' and (GetActionTexture(1) or GetActionTexture(109)) then
      PickupAction(1)
      PutItemInBackpack()
      PickupAction(109)
      PutItemInBackpack()
   end
   
   if NPB_class == 'druid' and (GetActionTexture(85) or GetActionTexture(109)) then
      PickupAction(85)
      PutItemInBackpack()
      PickupAction(109)
      PutItemInBackpack()
   end
   
   NumPadBar.cleared = true
end

----------------------------------------------------------------------------------------------------
-- checks for valid values before creating numpad frames
function NPB_CheckValues()
   local stances = NPB_CLASS_STANCEBARS[NPB_class]
   local n = NumPadBar
   if not n then
      n = { } end
   if not n.shape or not NPB_KEYBOARDS[n.shape] then
      n.shape = NPB_DEFAULT_SHAPE end
   if not n.enter then
      n.enter = '' end
   if not n.nav then
      n.nav = '' end
   if not n.arrow or NPB_class == 'moonkin' then
      n.arrow = '' end
   if not n.size or not NPB_SIZES[n.size] then
      n.size = NPB_DEFAULT_SIZE end
   if not n.left or n.left < 0 or n.left > 4000 then
      n.left = NPB_DEFAULT_LEFT end
   if not n.top or n.top < 0 or n.top > 4000 then
      n.top = NPB_DEFAULT_TOP end
   if not n.stancebars or not stances then
      n.stancebars = false end
   if not n.barpage or n.barpage < 1 or n.barpage > 6 then
      n.barpage = 1 end
   if n.labels ~= false then
      n.labels = true end
end

----------------------------------------------------------------------------------------------------
-- displays a message in the error frame (upper screen) for new users/characters
-- the message talks about left-clicking and right-clicking the numpad
function NPB_ClickMe(self, motion)
   if NumPadBar.leftclicked and NumPadBar.rightclicked then return end
   if UnitAffectingCombat('player') == 1 then return end
   if NPB_clickme then return end
   NPB_clickme = true
   NPB_Chat(NPB_DESCRIPTION)
   UIErrorsFrame:AddMessage(NPB_CLICK_ERROR, 1.0, 1.0, 1.0, 1.0, UIERRORS_HOLD_TIME)
end

----------------------------------------------------------------------------------------------------
-- shape of numpad dictates the key layout
-- makes draggable frame containing buttons, adds it to a global table of numpads
-- does not drag numpad or display drop down menu or change numpad if in combat
function NPB_CreateNumpad(shape, enter, nav, arrow)

   function NPB_DoDragStart(self, button)
      if UnitAffectingCombat('player') == 1 then return end
      NumPadBar.leftclicked = true
      NPB_SetBackdrop(self)
      self:StartMoving()
   end
   
   function NPB_DoDragStop(self, button)
      self:StopMovingOrSizing()
      NPB_ClearBackdrop(self)
      NumPadBar.left = self:GetLeft()     -- save these values so that numpad is placed in same position next login
      NumPadBar.top = self:GetTop()     -- must use getleft/top instead of getpoint due to changes in 2.2
   end     -- getleft/top are always positive offsets from bottomleft
   
   function NPB_DoMouseDown(self, button)
      if UnitAffectingCombat('player') == 1 then return end
      if IsMouseButtonDown('leftbutton') then
         NPB_ToggleBackdrop(self)
      else
         NPB_SetBackdrop(self)
         NumPadBar.rightclicked = true
         NPB_ToggleMenu()
      end
   end
   
   function NPB_DoEnter(self, motion)
      NPB_ClickMe(self, motion)
   end
   
   ----------------------------------------------------------------------------------------------------
   
   local numpad = CreateFrame('Frame', 'NumpadFrame', UIParent)
   numpad:SetMovable(1)
   numpad:EnableMouse(1)
   numpad:RegisterForDrag('LeftButton')
   numpad:SetScript('OnDragStart', NPB_DoDragStart)
   numpad:SetScript('OnDragStop', NPB_DoDragStop)
   numpad:SetScript('OnMouseDown', NPB_DoMouseDown)
   numpad:SetScript('OnEnter', NPB_DoEnter)
   numpad:SetWidth(NPB_DRAG_WIDTH)
   numpad:SetHeight(NPB_DRAG_HEIGHT)
   numpad:SetClampedToScreen(true)
   numpad:Hide()
   -- setpoint and setscale removed from this function since shownumpad always sets those
   
   local keyboard = NPB_KEYBOARDS[shape]
   for keynum, key in ipairs(keyboard) do
      local chosen = (key.type == enter or key.type == nav or key.type == arrow)
      if key.type == 'Numeric' or chosen then
         local keyname = shape .. enter .. nav .. arrow .. keynum -- unique name for each button
         NPB_CreateButton(numpad, keyname, keynum, key.col, key.row, key.width)
      end
   end
   
   NPB_numpads[shape .. enter .. nav .. arrow] = numpad
end 

----------------------------------------------------------------------------------------------------
-- Sets up all the button actions with and without stances
function NPB_SetButtonActions(button, keynum)
   local action = NPB_CLASS_SLOTS[NPB_class][keynum]

   if NumPadBar.stancebars then
      -- Generate event string for stances
      local output = ''
      for sx = 1, 3 do
         local stance = (keynum < 12) and NPB_STANCES[sx][keynum] or action
         output = output .. '[bonusbar:' .. sx .. ']' .. stance .. '; '
      end
      output = output .. action
   
      RegisterStateDriver(button, "page", output)
   else
      UnregisterStateDriver(button, "page")
      
      button:SetAttribute("action", action)
   end
end

----------------------------------------------------------------------------------------------------
-- creates a button based on key and attaches it to numpad
-- numpad is a frame, the parent of all the buttons to be created
-- keyname is unique, made from shape, enter pref, cursor pref, and keynum
-- keynum is the index of the key in the list of slots, 1 is first, 9 is ninth, 0 is tenth
-- col is the horizontal position of the key, 1 is leftmost (in drag area), 0 is one to the left, 2 is one to the right
-- row is the vertical position of the key, 1 is top (in drag area), 2 is one row down
function NPB_CreateButton(numpad, keyname, keynum, col, row)

   function NPB_OnDragStart(self, button)
      if UnitAffectingCombat('player') == 1 or LOCK_ACTIONBAR == '1' then return end
      PickupAction(ActionButton_GetPagedID(self))
   end
   
   local button = CreateFrame('CheckButton', keyname, numpad, 'ActionBarButtonTemplate')
   button:SetAttribute('type', 'action')

   SecureHandler_OnLoad(button)
   
   button:WrapScript(button, "OnAttributeChanged", [[
      name = name:match("^state%-(.+)")
      if name then
         control:RunAttribute("_onstate-"..name, name, value)
         return false
      end  ]]
   )
   
   button:SetAttribute("_onstate-page", [[
      local stateid, newstate = ...
      self:SetAttribute("action", tonumber(newstate)) ]]
   )
   
   NPB_SetButtonActions(button, keynum)
 
   local left = NPB_BUTTON_SIZE * (col - 1)
   local top = - (NPB_BUTTON_SIZE * (row - 1) + NPB_DRAG_HANDLE)
   button:SetAttribute('showgrid', 1)  -- display all buttons including empty buttons (buttons with no action)
   button:HookScript('OnAttributeChanged', ActionButton_UpdateFlash)   -- to fix flashing bug
   button:SetWidth(NPB_BUTTON_SIZE-2);
   button:SetHeight(NPB_BUTTON_SIZE-2); 
   button:SetScript('OnDragStart', NPB_OnDragStart)
   button:SetPoint('TOPLEFT', left, top)

   button:Show()   -- needed to show empty buttons even when showgrid=1, WoW bug?
   numpad:SetAttribute('addchild', button)
end

----------------------------------------------------------------------------------------------------
-- given shape, enter, cursor: get numpad
-- create it if it does not exist
function NPB_GetNumpad()
   local shape = NumPadBar.shape
   local enter = NumPadBar.enter
   local nav = NumPadBar.nav
   local arrow = NumPadBar.arrow
   if not shape then shape = 'Windows' end
   if not enter then enter = '' end
   if not nav then nav = '' end
   if not arrow then arrow = '' end
   local numpad = NPB_numpads[shape .. enter .. nav .. arrow]
   if numpad then return numpad end
   NPB_CreateNumpad(shape, enter, nav, arrow)
   local numpad = NPB_numpads[shape .. enter .. nav .. arrow]
   return numpad
end

----------------------------------------------------------------------------------------------------
function NPB_SetState()
   local numpad = NPB_GetNumpad()
   local keyboard = NPB_KEYBOARDS[NumPadBar.shape]
   local enter = NumPadBar.enter
   local nav = NumPadBar.nav
   local arrow = NumPadBar.arrow

   for keynum, key in ipairs(keyboard) do
      local chosen = (key.type == enter or key.type == nav or key.type == arrow)
      if key.type == 'Numeric' or chosen then
         local name = NumPadBar.shape .. enter .. nav .. arrow .. keynum
         local button = _G[name]
         
         NPB_SetButtonActions(button, keynum)
      end
   end
end

----------------------------------------------------------------------------------------------------
function NPB_ShowNumpad()
   local numpad = NPB_GetNumpad()
   local keyboard = NPB_KEYBOARDS[NumPadBar.shape]
   local enter = NumPadBar.enter
   local nav = NumPadBar.nav
   local arrow = NumPadBar.arrow
   ClearOverrideBindings(numpad)
   for keynum, key in ipairs(keyboard) do
      local chosen = (key.type == enter or key.type == nav or key.type == arrow)
      if key.type == 'Numeric' or chosen then
         local name = NumPadBar.shape .. enter .. nav .. arrow .. keynum
            -- this is the unique name of the button
         local label = (NumPadBar.labels) and key.label or ''
         local button = _G[name]
         button.npb_pad = true
         button.npb_label = label
         NPB_DisplayHotkey(button)
   
         SetOverrideBindingClick(numpad, false, key.binding, name)
         SetOverrideBindingClick(numpad, false, 'Shift-' .. key.binding, name)
         -- key.binding is the key binding such as A or Enter or NumpadDecimal
         -- also disables the default WoW key bindings for shift-numpad functions
         -- especially Previous Action Bar and Next Action Bar
         -- so that your bottom bar is not accidentally changed in combat
         -- SetOverrideBindingClick(owner, isPriority, 'KEY', 'buttonName' [,'mouseButton'])
         -- only works when the button is showing (according to blizzard)
      end
   end
   numpad:SetScale(NumPadBar.size)
   numpad:ClearAllPoints()
   numpad:SetPoint('TOPLEFT', UIParent, 'BOTTOMLEFT', NumPadBar.left, NumPadBar.top)
      -- was changed for 2.2, left and top are always positive offsets from bottomleft
   
   if not NumPadBar.leftclicked or not NumPadBar.rightclicked then
      NPB_SetBackdrop(numpad) end   -- if user has not moved or used the menu, display the background
   --NPB_SetState()
   if NumPadBar.hidden then return end
   numpad:Show()
   NumPadBar.disabled = false
end

----------------------------------------------------------------------------------------------------
function NPB_HideNumpad()
   local numpad = NPB_GetNumpad()
   ClearOverrideBindings(numpad)
   numpad:Hide()
   NumPadBar.disabled = true
end

----------------------------------------------------------------------------------------------------
function NPB_ShapeNumpad(shape)
   if UnitAffectingCombat('player') == 1 then return end
   if not NumPadBar.shape or not NPB_KEYBOARDS[NumPadBar.shape] then
      NumPadBar.shape = 'Windows' end
   if not shape then  -- if not specified, then next in list after current shape
      shape = next(NPB_KEYBOARDS, NumPadBar.shape) end
   if not shape then  -- if not specified and nothing next in list, use first in list
      shape = next(NPB_KEYBOARDS, nil) end
   NPB_HideNumpad()
   NumPadBar.shape = shape
   NPB_ShowNumpad()
end

----------------------------------------------------------------------------------------------------
function NPB_SizeNumpad(size)
   if UnitAffectingCombat('player') == 1 then return end
   if not NumPadBar.size or not NPB_SIZES[NumPadBar.size] then
      NumPadBar.size = 1 end
   if not size then
      size = NPB_SIZES[NumPadBar.size] end
   if not size then
      size = 1 end
   NumPadBar.left = NumPadBar.left * NumPadBar.size / size
   NumPadBar.top = NumPadBar.top * NumPadBar.size / size
   NumPadBar.size = size
   NPB_ShowNumpad()
end

----------------------------------------------------------------------------------------------------
function NPB_ToggleEnter()
   if UnitAffectingCombat('player') == 1 then return end
   NPB_HideNumpad()
   NumPadBar.enter = (NumPadBar.enter == 'Enter') and '' or 'Enter'
   if NumPadBar.enter == 'Enter' then
      UIErrorsFrame:AddMessage(NPB_ENTER_UNBOUND, 1.0, 1.0, 1.0, 1.0, UIERRORS_HOLD_TIME) end
   NPB_ShowNumpad()
end

----------------------------------------------------------------------------------------------------
function NPB_ToggleNav()
   if UnitAffectingCombat('player') == 1 then return end
   NPB_HideNumpad()
   NumPadBar.nav = (NumPadBar.nav == 'Nav') and '' or 'Nav'
   if NumPadBar.nav == 'Nav' then
      UIErrorsFrame:AddMessage(NPB_NAV_UNBOUND, 1.0, 1.0, 1.0, 1.0, UIERRORS_HOLD_TIME) end
   NPB_ShowNumpad()
end

----------------------------------------------------------------------------------------------------
function NPB_ToggleArrow()
   if UnitAffectingCombat('player') == 1 then return end
   NPB_HideNumpad()
   NumPadBar.arrow = (NumPadBar.arrow == 'Arrow') and '' or 'Arrow'
   if NumPadBar.arrow == 'Arrow' and NPB_class == 'moonkin' then
      NPB_Chat(NPB_ARROW_TOPSLOTS) end
   if NumPadBar.arrow == 'Arrow' and (NPB_class == 'druid' or NPB_class == 'warrior') then
      NPB_Chat(NPB_ARROW_RIGHTSLOTS) end
   if NumPadBar.arrow == 'Arrow' then
      UIErrorsFrame:AddMessage(NPB_ARROW_UNBOUND, 1.0, 1.0, 1.0, 1.0, UIERRORS_HOLD_TIME) end
   NPB_ShowNumpad()
end

----------------------------------------------------------------------------------------------------
function NPB_ToggleLabels()
   if UnitAffectingCombat('player') == 1 then return end
   NPB_HideNumpad()
   NumPadBar.labels = not NumPadBar.labels
   NPB_ShowNumpad()
end

----------------------------------------------------------------------------------------------------
function NPB_ToggleStancebars()
   if UnitAffectingCombat('player') == 1 then return end
   local stances = NPB_CLASS_STANCEBARS[NPB_class]
   if not stances then NPB_Chat(NPB_STANCELESS_CLASS) return end
   NumPadBar.stancebars = not NumPadBar.stancebars
   NPB_SetState()
   if NumPadBar.stancebars then
      NPB_Chat(NPB_USING_STANCEBARS)
   else
      NPB_Chat(NPB_NOT_USING_STANCEBARS)
      if NPB_class == 'moonkin' then NPB_Chat(NPB_MOONKIN_BAR) end
   end
end

----------------------------------------------------------------------------------------------------
function NPB_Disable()
   if UnitAffectingCombat('player') == 1 then return end
   NPB_HideNumpad()
   NPB_Chat(NPB_DISABLED)
end

----------------------------------------------------------------------------------------------------
function NPB_Hide()
   local numpad = NPB_GetNumpad()
   numpad:Hide()
   NumPadBar.hidden = true
   NPB_Chat(NPB_HIDDEN)
end

function NPB_Show()
   local numpad = NPB_GetNumpad()
   NumPadBar.hidden = false
   numpad:Show()
end

----------------------------------------------------------------------------------------------------
function NPB_Help()
   if UnitAffectingCombat('player') == 1 then return end
   ScrollTip(NPB_INSTRUCTIONS, 'large')
end

----------------------------------------------------------------------------------------------------
-- BACKDROP FUNCTIONS
----------------------------------------------------------------------------------------------------

function NPB_SetBackdrop(frame)
   NPB_backdrop = true
   frame:SetBackdrop(NPB_BACKDROP)
end

function NPB_ClearBackdrop(frame)
   NPB_backdrop = false
   frame:SetBackdrop(nil)
end

function NPB_ToggleBackdrop(frame)
   if not NPB_backdrop then
      NPB_SetBackdrop(frame)
   else
      NPB_ClearBackdrop(frame)
   end
end

----------------------------------------------------------------------------------------------------
-- DROP DOWN MENU FUNCTIONS
----------------------------------------------------------------------------------------------------

function NPB_InitializeMenu()
   local s = NumPadBar.shape
   local menu = {
      { text = NPB_MENU.windows, func = NPBM_windows, checked = (s == 'Windows') },
      { text = NPB_MENU.office, func = NPBM_office, checked = (s == 'MicrosoftOffice') },
      { text = NPB_MENU.multimedia, func = NPBM_multimedia, checked = (s == 'NaturalMultimedia') },
      { text = NPB_MENU.elite, func = NPBM_elite, checked = (s == 'NaturalElite') },
      { text = NPB_MENU.macintosh, func = NPBM_macintosh, checked = (s == 'Macintosh') },
      { text = NPB_MENU.large, func = NPBM_large, checked = (NumPadBar.size == 1) },
      { text = NPB_MENU.medium, func = NPBM_medium, checked = (NumPadBar.size == .8) },
      { text = NPB_MENU.small, func = NPBM_small, checked = (NumPadBar.size == .67) },
      { text = NPB_MENU.enter, func = NPBM_enter, checked = (NumPadBar.enter == 'Enter') },
      { text = NPB_MENU.nav, func = NPBM_nav, checked = (NumPadBar.nav == 'Nav') },
      { text = NPB_MENU.arrow, func = NPBM_arrow, checked = (NumPadBar.arrow == 'Arrow') },
      { text = NPB_MENU.labels, func = NPBM_labels, checked = NumPadBar.labels },
      { text = NPB_MENU.stancebars, func = NPBM_stancebars, checked = NumPadBar.stancebars },
      { text = NPB_MENU.disable, func = NPBM_disable },
      { text = NPB_MENU.hide, func = NPBM_hide },
      { text = NPB_MENU.help, func = NPBM_help }
   }
   for ix in ipairs(menu) do
      UIDropDownMenu_AddButton(menu[ix]) 
   end
end

function NPB_ToggleMenu()
   ToggleDropDownMenu(1, nil, NPB_MenuFrame, 'cursor')
end

function NPB_DoneMenu()
   if UnitAffectingCombat('player') == 1 then return end
   local numpad = NPB_GetNumpad()
   NPB_ClearBackdrop(numpad)
end

function NPBM_windows()    NPB_ShapeNumpad('Windows'); NPB_DoneMenu(); end
function NPBM_office()     NPB_ShapeNumpad('MicrosoftOffice'); NPB_DoneMenu(); end
function NPBM_multimedia() NPB_ShapeNumpad('NaturalMultimedia'); NPB_DoneMenu(); end
function NPBM_elite()      NPB_ShapeNumpad('NaturalElite'); NPB_DoneMenu(); end
function NPBM_macintosh()  NPB_ShapeNumpad('Macintosh'); NPB_DoneMenu(); end
function NPBM_large()      NPB_SizeNumpad(1); NPB_DoneMenu(); end
function NPBM_medium()     NPB_SizeNumpad(.8); NPB_DoneMenu(); end
function NPBM_small()      NPB_SizeNumpad(.67); NPB_DoneMenu(); end
function NPBM_enter()      NPB_ToggleEnter(); NPB_DoneMenu(); end
function NPBM_nav()        NPB_ToggleNav(); NPB_DoneMenu(); end
function NPBM_arrow()      NPB_ToggleArrow(); NPB_DoneMenu(); end
function NPBM_labels()     NPB_ToggleLabels(); NPB_DoneMenu(); end
function NPBM_stancebars() NPB_ToggleStancebars(); NPB_DoneMenu(); end
function NPBM_disable()    NPB_Disable(); NPB_DoneMenu(); end
function NPBM_help()       NPB_Help(); NPB_DoneMenu(); end
function NPBM_hide()       NPB_Hide(); NPB_DoneMenu(); end