--[[
Copyright 2007, 2008, 2009, 2010 João Cardoso
MailNotifier is distributed under the terms of the GNU General Public License (or the Lesser GPL).
This file is part of MailNotifier.

MailNotifier is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

MailNotifier is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with MailNotifier. If not, see <http://www.gnu.org/licenses/>.
--]]

local MailNotifier = CreateFrame('Frame', 'MailNotifier')
local Indicator, Shine, Timer, FakeEvent, NewMails, AtInbox
local L = MailNotifier_Locals

local AUCTION_OUTBID = ERR_AUCTION_OUTBID_S:gsub('%%s', '%.+')
local AUCTION_WON = ERR_AUCTION_WON_S:gsub('%%s', '%.+')


--[[ Startup ]]--

function MailNotifier:Startup()
	Indicator = MiniMapMailFrame:CreateFontString('MailNotifierIndicator', 'OVERLAY')
	Indicator:SetFontObject('NumberFontNormal')
	Indicator:SetPoint('Center', MiniMapMailFrame, 'Center', -1, 1)
	Indicator:SetJustifyH('CENTER')

	Shine = MiniMapMailFrame:CreateTexture('MailNotifierShine', 'OVERLAY')
	Shine:SetTexture('Interface\\Calendar\\EventNotificationGlow')
	Shine:SetPoint('CENTER', MiniMapMailFrame, 'CENTER', -2, 2)
	Shine:SetHeight(35) Shine:SetWidth(35)
	Shine:SetBlendMode('ADD')
	Shine:Hide()
	
	self:SetScript('OnEvent', function(self, event, ...) self[event](self, ...) end)
	self:RegisterEvent('VARIABLES_LOADED')
	self:RegisterEvent('PLAYER_LOGOUT')
	self.Startup = nil
end

function MailNotifier:VARIABLES_LOADED()
	MailNotifier_Senders = MailNotifier_Senders or {}
	MailNotifier_Count = MailNotifier_Count or 0
	
	function self:UPDATE_PENDING_MAIL()
		GetLatestThreeSenders()
		Timer = 0
	end
	
	self:UPDATE_PENDING_MAIL()
	self.VARIABLES_LOADED = nil
	self:RegisterEvent('UPDATE_PENDING_MAIL')
	self:SetScript('OnUpdate', function(self, elapsed)
		Timer = Timer + elapsed
		if Timer >= 3 then
			self:RegisterEvent('PLAYER_ENTERING_WORLD')
			self:RegisterEvent('MAIL_INBOX_UPDATE')
			self:RegisterEvent('CHAT_MSG_SYSTEM')
			self:RegisterEvent('MAIL_CLOSED')
			self:SetScript('OnUpdate', nil)
			
			local newMails, oldI, off = 0
			for newI, sender in pairs({GetLatestThreeSenders()}) do
				oldI = MailNotifier_Senders[sender] or 0
					
				if newI >= oldI then
					off = newI - oldI
				else
					off = 3 - oldI + newI
				end
					
				newMails = max(newMails, off)
			end
			self:AddNewMail(newMails)
			
			function self:UPDATE_PENDING_MAIL()
				if FakeEvent or AtInbox then
					FakeEvent = nil
				else
					self:AddNewMail(1)
				end
			end
		end
	end)
end


--[[ Inbox ]]--

function MailNotifier:MAIL_INBOX_UPDATE()
	local newMails = 0
	for i = 1, GetInboxNumItems() do
		if not select(9, GetInboxHeaderInfo(i)) then
			newMails = newMails + 1
		end
	end
	self:SetNumMails(newMails)
	
	if not AtInbox then
		NewMails = self:GetNumMails()
		AtInbox = true
	elseif self:GetNumMails() ~= NewMails then
		FakeEvent = true
	end
end

function MailNotifier:MAIL_CLOSED()
	AtInbox = nil
end


--[[ Other Events ]]--

function MailNotifier:PLAYER_ENTERING_WORLD()
	FakeEvent = true
end

function MailNotifier:CHAT_MSG_SYSTEM(message)
	if strmatch(message, AUCTION_OUTBID) then
		self:UPDATE_PENDING_MAIL()
	elseif arg1 == ERR_AUCTION_REMOVED or strmatch(message, AUCTION_WON) then
		FakeEvent = true
	end
end  

function MailNotifier:PLAYER_LOGOUT()
	local s1, s2, s3 = GetLatestThreeSenders()
	MailNotifier_Senders = {
		[s1 or ''] = 1,
		[s2 or ''] = 2,
		[s3 or ''] = 3,
	}
end


--[[ API ]]--

function MailNotifier:AddNewMail(new)
	self:SetNumMails(self:GetNumMails() + new)
	if new > 0 then
		PlaySoundFile('Interface\\AddOns\\MailNotifier\\NewMail.wav')
		UIFrameFlash(Shine, 1, 1, 10, nil, 0.6)
	end
end

function MailNotifier:SetNumMails(value)
	MailNotifier_Count = max(value or 0, HasNewMail() and 1 or 0 , select('#', GetLatestThreeSenders()))
	Indicator:SetText(MailNotifier_Count or '')
end

function MailNotifier:GetNumMails()
	return MailNotifier_Count
end


--[[ Tootlip ]]--

function MinimapMailFrameUpdate()
	local numMails = MailNotifier:GetNumMails()
	local senders = {GetLatestThreeSenders()}
	local title
	
	if numMails ~= 1 then
		title = L.HaveMails
	else
		title = L.HaveMail
	end
			
	if #senders > 0 then
		title = title .. L.From
	end
	
	GameTooltip:SetText(format(title, numMails))
	
	for i,sender in pairs(senders) do
		GameTooltip:AddLine(' - ' .. sender, 1, 1, 1)
	end
	
	GameTooltip:Show()
end


--[[ Start Addon ]]--

MailNotifier:Startup()