if select(6, GetAddOnInfo("Cartographer3_" .. (debugstack():match("[a%.][p%.][h%.]er3\\(.-)\\") or ""))) ~= "MISSING" then return end

local Cartographer3 = _G.Cartographer3
if not Cartographer3 then
	error("Cartographer3_Waypoints requires Cartographer3")
end

Cartographer3.L("Waypoints", "enUS", function() return {
} end)

local L = Cartographer3.L("Waypoints")

local Cartographer3_Waypoints = Cartographer3.NewModule("Waypoints", L["Waypoints"], L["Module which shows an arrow to direct you to a specified location"], {
	arrowSize = 1,
	position = { "TOP", 0, -50 },
	questHelper = true,
})

local WotLK = not not ToggleAchievementFrame

local Cartographer3_Utils = Cartographer3.Utils

local db

local updateArrow

local waypoint_x, waypoint_y, waypoint_text, waypoint_metadata, waypoint_unit
local next_waypoint_x, next_waypoint_y, next_waypoint_text, next_waypoint_metadata, next_waypoint_unit

local timeSinceLastSetWaypoint = 0

local arrow

function Cartographer3_Waypoints.SetWaypoint(...)
	if not Cartographer3_Waypoints.IsEnabled() then
		return
	end
	local first = (...)
	local zone, x, y, text, metadata
	if type(first) == "number" then
		local continentID, zoneID
		continentID, zoneID, x, y, text, metadata = ...
		zone = Cartographer3.Utils.ConvertCIDAndZIDToMapTexture(continentID, zoneID)
		if not zone then
			return
		end
	else
		zone, x, y, text, metadata = ...
	end
	local ux, uy = Cartographer3_Utils.ConvertZoneCoordinateToUniverseCoordinate(zone, x, y)
	
	local currentTime = GetTime()
	
	if timeSinceLastSetWaypoint + 1 > currentTime then
		-- annoyingly quick
		next_waypoint_x = ux
		next_waypoint_y = uy
		next_waypoint_text = text
		next_waypoint_metadata = metadata
		next_waypoint_unit = nil
	else
		waypoint_x = ux
		waypoint_y = uy
		waypoint_text = text
		waypoint_metadata = metadata
		waypoint_unit = nil
		timeSinceLastSetWaypoint = currentTime
	end
	
	updateArrow(0, GetTime())
end
_G.SetWaypoint = Cartographer3_Waypoints.SetWaypoint

function Cartographer3_Waypoints.GetWaypoint()
	return waypoint_metadata
end
_G.GetWaypoint = Cartographer3_Waypoints.GetWaypoint

function Cartographer3_Waypoints.ClearWaypoint()
	waypoint_x = nil
	waypoint_y = nil
	waypoint_text = nil
	waypoint_metadata = nil
	waypoint_unit = nil
	next_waypoint_x = nil
	next_waypoint_y = nil
	next_waypoint_text = nil
	next_waypoint_metadata = nil
	next_waypoint_unit = nil
	
	updateArrow(0, GetTime())
end
_G.ClearWaypoint = Cartographer3_Waypoints.ClearWaypoint

function Cartographer3_Waypoints.SetWaypointToUnit(unit)
	if not Cartographer3_Waypoints.IsEnabled() then
		return
	end
	local name, server = UnitName(unit)
	local _, class = UnitClass(unit)
	local classColor = RAID_CLASS_COLORS[class]
	if classColor then
		name = ("|cff%02x%02x%02x%s|r"):format(classColor.r * 255, classColor.g * 255, classColor.b * 255, name)
	end
	if server and server ~= "" then
		name = name .. "-" .. server
	end
	
	local level = UnitLevel(unit)
	local difficultyColor = GetDifficultyColor(level)
	if difficultyColor then
		name = ("[|cff%02x%02x%02x%d|r] %s"):format(difficultyColor.r * 255, difficultyColor.g * 255, difficultyColor.b * 255, level, name)
	end
	local currentTime = GetTime()
	if timeSinceLastSetWaypoint + 1 > currentTime then
		-- annoyingly quick
		next_waypoint_unit = unit
		next_waypoint_text = name
		next_waypoint_metadata = "unit"
		next_waypoint_x = nil
		next_waypoint_y = nil
	else
		waypoint_unit = unit
		waypoint_text = name
		waypoint_metadata = "unit"
		waypoint_x = nil
		waypoint_y = nil
		timeSinceLastSetWaypoint = currentTime
	end
	
	updateArrow(0, GetTime())
end

function Cartographer3_Waypoints.Disable()
	Cartographer3_Waypoints.ClearWaypoint()
end

do
	local function getPOIUniversePosition(poi)
		local numPoints = poi:GetNumPoints()
		if numPoints ~= 1 then
			return
		end
		local point, attach, relpoint, ux, uy = poi:GetPoint(1)
		if point == "CENTER" and relpoint == "CENTER" and attach == Cartographer3.Data.mapView then
			return ux, uy
		end
	end
	local zoneTexture, zoneX, zoneY
	local function setWaypointHere()
		Cartographer3_Waypoints.SetWaypoint(zoneTexture, zoneX, zoneY)
	end
	local function setWaypointToPOI(poi)
		local ux, uy = getPOIUniversePosition(poi)
		local zone, x, y = Cartographer3_Utils.ConvertUniverseCoordinateToZoneCoordinate(ux, uy)
		local text = poi.name
		if poi.description and poi.description:trim() ~= "" then
			if text then
				text = text .. " - " .. poi.description:trim()
			else
				text = poi.description:trim()
			end
		end
		Cartographer3_Waypoints.SetWaypoint(zone, x, y, text, "POI")
	end
	Cartographer3.AddRightClickMenuHandler(function(data, level, value, needSeparator)
		if not Cartographer3_Waypoints.IsEnabled() then
			return
		end
		if not data.zoneData and not data.continentData and not data.battlegroundFrame then
			return
		end
		if level == 1 then
			if needSeparator then
				local info = UIDropDownMenu_CreateInfo()
				info.text = " "
				info.isTitle = true
				UIDropDownMenu_AddButton(info, level)
			end
			local info = UIDropDownMenu_CreateInfo()
			info.text = L["Set waypoint here"]
			info.func = setWaypointHere
			zoneTexture, zoneX, zoneY = data.zoneTexture, data.zoneX, data.zoneY
			UIDropDownMenu_AddButton(info, level)
			
			for i, poi in ipairs(data.pois) do
				local unit = poi.unit
				if unit and unit ~= "player" then
					local name, server = UnitName(unit)
					local _, class = UnitClass(unit)
					local classColor = RAID_CLASS_COLORS[class]
					if classColor then
						name = ("|cff%02x%02x%02x%s|r"):format(classColor.r * 255, classColor.g * 255, classColor.b * 255, name)
					end
					if server and server ~= "" then
						name = name .. "-" .. server
					end
					
					local info = UIDropDownMenu_CreateInfo()
					info.text = L["Set waypoint to %s"]:format(name)
					info.func = Cartographer3_Waypoints.SetWaypointToUnit
					info.arg1 = unit
					UIDropDownMenu_AddButton(info, level)
				else
					local name = poi.name
					if name then
						local ux, uy = getPOIUniversePosition(poi)
						if ux then
							local info = UIDropDownMenu_CreateInfo()
							info.text = L["Set waypoint to %s"]:format(name)
							info.func = setWaypointToPOI
							info.arg1 = poi
							UIDropDownMenu_AddButton(info, level)
						end
					end
				end
			end
			return true
		end
	end)
end

function updateArrow()
	assert(db)
	local button = CreateFrame("Button", "Cartographer3_Waypoints_Button", UIParent)
	button:SetWidth(100)
	button:SetHeight(55)
	button:SetPoint(db.position[1], UIParent, db.position[1], db.position[2], db.position[3])
	button:EnableMouse(true)
	button:SetMovable(true)
	button:RegisterForDrag("LeftButton")
	button:SetScript("OnDragStart", function(this)
		this:StartMoving()
	end)
	button:SetScript("OnDragStop", function(this)
		this:StopMovingOrSizing()
		
		local width, height = GetScreenWidth(), GetScreenHeight()
		local uiscale = UIParent:GetEffectiveScale()
		local scale = this:GetEffectiveScale() / uiscale
		local x, y = this:GetCenter()
		x, y = x*scale, y*scale
		local point
		if x < width/3 then
			point = "LEFT"
			x = x - this:GetWidth()/2*scale
			if x < 0 then
				x = 0
			end
		elseif x < width*2/3 then	
			point = ""
			x = x - width/2
		else
			point = "RIGHT"
			x = x - width + this:GetWidth()/2*scale
			if x > 0 then
				x = 0
			end
		end
		if y < height/3 then
			point = "BOTTOM" .. point
			y = y - this:GetHeight()/2*scale
			if y < 0 then
				y = 0
			end
		elseif y < height*2/3 then
			if point == "" then
				point = "CENTER"
			end
			y = y - height/2
		else
			point = "TOP" .. point
			y = y - height + this:GetHeight()/2*scale
			if y > 0 then
				y = 0
			end
		end
		x, y = x/scale, y/scale
		this:ClearAllPoints()
		this:SetPoint(point, UIParent, point, x, y)
		db.position[1] = point
		db.position[2] = x
		db.position[3] = y
	end)
	button:RegisterForClicks("RightButtonUp")
	button:Hide()
	button:SetClampedToScreen(true)
	local dropdownFrame
	local menuList
	local locked = false
	button:SetScript("OnClick", function(this)
		if not dropdownFrame then
			dropdownFrame = CreateFrame("Frame", this:GetName() .. "_Dropdown", this, "UIDropDownMenuTemplate")
			-- Clear waypoint
			-- Lock arrow
			menuList = {
				{
					text = L["Cartographer3 Waypoint Arrow"],
					isTitle = true,
				},
				{
					text = L["Clear waypoint"],
					func = Cartographer3_Waypoints.ClearWaypoint,
				},
				{
					text = L["Lock arrow position"],
					func = function()
						locked = not locked
						if locked then
							button:SetMovable(false)
							button:RegisterForDrag()
						else
							button:SetMovable(true)
							button:RegisterForDrag("LeftButton")
						end
						menuList[3].checked = locked
					end,
					checked = locked
				},
				{
					text = " ",
					isTitle = true,
				},
				{
					text = L["Close menu"],
					func = function() end,
				}
			}
		end
		EasyMenu(menuList, dropdownFrame, "cursor", 0, 0, "MENU")
	end)
	
	local path = debugstack(1, 1, 0)
	if path:find("grapher3\\") then
		path = [=[Interface\AddOns\Cartographer3\Waypoints\WaypointArrow.mdx]=]
	elseif path:find("grapher3_Waypoints\\") then
		path = [=[Interface\AddOns\Cartographer3_Waypoints\WaypointArrow.mdx]=]
	else
		error("Cannot determine path of artwork")
	end
	
	arrow = CreateFrame("PlayerModel", nil, button)
	arrow:SetModel(path)
	arrow:SetModelScale(1.55)
	arrow:SetLight(1, 0, 0, 1, 0, 1, 0.7, 0.7, 0.7, 1, 0.8, 0.8, 0.64)
	arrow:SetWidth(100)
	arrow:SetHeight(55)
	arrow:SetPoint("CENTER")
	
	
	local text = button:CreateFontString(nil, "ARTWORK", "GameFontHighlight")
	button.text = text
	text:SetPoint("TOP", arrow, "BOTTOM")
	
	local yardage = button:CreateFontString(nil, "ARTWORK", "GameFontNormal")
	button.yardage = yardage
	yardage:SetPoint("TOP", text, "BOTTOM")
	
	local time = button:CreateFontString(nil, "ARTWORK", "GameFontNormal")
	button.time = time
	time:SetPoint("TOP", yardage, "BOTTOM")
	
	local YARDS_PER_PIXEL = Cartographer3.Data.YARDS_PER_PIXEL
	
	local function resetModel()
		arrow:SetModel(path)
		local model = arrow:GetModel()
		if not model or not model.match or not model:match("[Aa]rrow") then
			arrow:Hide()
			arrow:ClearAllPoints()
			arrow:SetParent(UIParent)
			arrow = CreateFrame("PlayerModel", nil, button)
			arrow:SetModel(path)
			arrow:SetModelScale(1.55 / button:GetScale())
			arrow:SetLight(1, 0, 0, 1, 0, 1, 0.7, 0.7, 0.7, 1, 0.8, 0.8, 0.64)
			arrow:SetWidth(100)
			arrow:SetHeight(55)
			arrow:SetPoint("CENTER")
			text:SetPoint("TOP", arrow, "BOTTOM")
		end
	end
	
	local last_playerX, last_playerY
	local lastUpdate = 0
	function updateArrow(elapsed, currentTime)
		if (next_waypoint_x or next_waypoint_unit) and timeSinceLastSetWaypoint + 1 < currentTime then
			waypoint_x = next_waypoint_x
			waypoint_y = next_waypoint_y
			waypoint_unit = next_waypoint_unit
			waypoint_text = next_waypoint_text
			waypoint_metadata = next_waypoint_metadata
			next_waypoint_x, next_waypoint_y, next_waypoint_unit, next_waypoint_text, next_waypoint_metadata = nil, nil, nil, nil, nil
			timeSinceLastSetWaypoint = currentTime
		end
		if waypoint_unit then
			waypoint_x, waypoint_y = Cartographer3.Utils.GetUnitUniverseCoordinate(waypoint_unit)
		end
		
		if not waypoint_x then
			button:Hide()
			return
		end
		
		local playerX, playerY = Cartographer3.Utils.GetUnitUniverseCoordinate("player")
		if not playerX then
			button:Hide()
			return
		end
		
		local destinationDeltaX, destinationDeltaY = waypoint_x - playerX, waypoint_y - playerY
		
		local angleToDestination = math.atan2(destinationDeltaY, destinationDeltaX)
		
		local playerRotation = Cartographer3.Utils.GetPlayerRotation()
		
		local arrowRotation = angleToDestination + playerRotation
		arrow:SetRotation(arrowRotation)
		
		local value = (((arrowRotation / math.pi) % 2) - 1)^2
		
		if value >= 0.5 then
			r, g, b = 2 - value*2, 1, 0
		else
			r, g, b = 1, value*2, 0
		end
		arrow:SetLight(1, 0, 0, 0.2, -0.8, 0.5, r, g, b, 0.8, r, g, b)
		
		button.text:SetText(waypoint_text)
		
		local yardsToDestination = (destinationDeltaX*destinationDeltaX + destinationDeltaY*destinationDeltaY)^0.5 * YARDS_PER_PIXEL
		
		if yardsToDestination < 5 then -- allow changing of yards
			Cartographer3_Waypoints.ClearWaypoint()
			return
		end
		
		if not button:IsShown() then
			button:Show()
			resetModel()
		else
			local model = arrow:GetModel()
			if not model or not model.match or not model:match("[Aa]rrow") then
				arrow:SetModel([=[Character\NightElf\Female\NightElfFemale.mdx]=])
				button:Hide()
			end
		end
		
		local yardString = L["%.0f yd"]:format(yardsToDestination)
		button.yardage:SetText(yardString)
		
		local diffTime = currentTime - lastUpdate
		if diffTime >= 1 then
			lastUpdate = currentTime
			if not last_playerX then
				last_playerX, last_playerY = playerX, playerY
				return
			end
			
			local movementX, movementY = playerX - last_playerX, playerY - last_playerY
			local movementAngle = math.atan2(movementY, movementX)
			local movementYardsPerSecond = (movementX*movementX + movementY*movementY)^0.5 * YARDS_PER_PIXEL / diffTime
			movementYardsPerSecond = math.floor(movementYardsPerSecond * 100 + 0.5) / 100
			
			local secondsToDestination = yardsToDestination / movementYardsPerSecond
			
			local adjustedSecondsToDestination = secondsToDestination / math.cos(movementAngle - angleToDestination)
			
			local timeString
			if adjustedSecondsToDestination < 0 or adjustedSecondsToDestination > 24*60*60 then
				timeString = '---'
			else
				if adjustedSecondsToDestination > 60*60 then
					timeString = ("%d:%02d:%02d"):format(adjustedSecondsToDestination / (60*60), (adjustedSecondsToDestination % (60*60)) / 60, adjustedSecondsToDestination % 60)
				else
					timeString = ("%d:%02d"):format(adjustedSecondsToDestination / 60, adjustedSecondsToDestination % 60)
				end
			end
		
			button.time:SetText(timeString)
			last_playerX, last_playerY = playerX, playerY
		end
	end
	Cartographer3_Utils.AddTimer(updateArrow, true)
	Cartographer3_Waypoints.SetArrowSize(db.arrowSize)
	updateArrow(0, GetTime())
end

function Cartographer3_Waypoints.SetArrowSize(value)
	db.arrowSize = value
	if _G.Cartographer3_Waypoints_Button then
		_G.Cartographer3_Waypoints_Button:SetScale(value)
		arrow:SetModelScale(1.55 / value)
	end
end

function Cartographer3_Waypoints.OnInitialize()
	db = Cartographer3.db.Waypoints
	Cartographer3_Waypoints.SetArrowSize(db.arrowSize)
end

local LibSimpleOptions = LibStub("LibSimpleOptions-1.0")

LibSimpleOptions.AddSuboptionsPanel("Cartographer3", L["Waypoints"], function(self)
	local title, subText = self:MakeTitleTextAndSubText(L["Waypoints"], L["These options allow you to configure Cartographer3's waypoints"])
	
	local enableToggle = self:MakeToggle(
		'name', L["Enabled"],
		'description', L["Enable or disable the Waypoints module"],
		'default', true,
		'getFunc', Cartographer3_Waypoints.IsEnabled,
		'setFunc', function(value)
			if value then
				Cartographer3.EnableModule("Waypoints")
			else
				Cartographer3.DisableModule("Waypoints")
			end
		end)
	enableToggle:SetPoint("TOPLEFT", subText, "BOTTOMLEFT", 0, -24)
	
	local arrowSizeSlider = self:MakeSlider(
		'name', L["Arrow Size"],
		'description', L["The size of the waypoint arrow"],
		'minText', "25%",
		'maxText', "400%",
		'minValue', 0.25,
		'maxValue', 4,
		'step', 0.05,
		'default', 1,
		'current', db.arrowSize,
		'setFunc', function(value)
			Cartographer3_Waypoints.SetArrowSize(value)
		end,
		'currentTextFunc', function(value)
			return ("%.0f%%"):format(value * 100)
		end)
	arrowSizeSlider:SetPoint("TOPLEFT", enableToggle, "BOTTOMLEFT", 0, -24)
	
	if _G.QuestHelper then
		local questHelperToggle = self:MakeToggle(
			'name', L["Show QuestHelper waypoints"],
			'description', L["Whether to show waypoints dictated by QuestHelper"],
			'default', true,
			'current', db.questHelper,
			'setFunc', function(value)
				db.questHelper = value
				if not value and waypoint_metadata == "QuestHelper" then
					Cartographer3_Waypoints.ClearWaypoint()
				end
			end)
		questHelperToggle:SetPoint("TOPLEFT", arrowSizeSlider, "BOTTOMLEFT", 0, -24)
	end
end)


for i, v in ipairs { "/waypoint", "/way", "/tway", "/tloc" } do
	_G["SLASH_CARTOGRAPHERTHREE_WAYPOINTS" .. i] = v
end

local function printUsage()
	DEFAULT_CHAT_FRAME:AddMessage(L["|cffffff7fCartographer3 - Waypoints |r/way |cffffff7fUsage:|r"])
	DEFAULT_CHAT_FRAME:AddMessage(L["|cffffff7f/way <x> <y> [description]|r - Set waypoint to x, y with description"])
	DEFAULT_CHAT_FRAME:AddMessage(L["|cffffff7f/way <zone> <x> <y> [description]|r - Set waypoint to x, y in zone with description"])
	DEFAULT_CHAT_FRAME:AddMessage(L["|cffffff7f/way clear|r - Clear waypoint"])
end

local zoneList
local possibleMatches = {}
_G.hash_SlashCmdList.CARTOGRAPHERTHREE_WAYPOINTS = nil
_G.SlashCmdList.CARTOGRAPHERTHREE_WAYPOINTS = function(text)
	if not Cartographer3_Waypoints.IsEnabled() then
		DEFAULT_CHAT_FRAME:AddMessage(L["|cffffff7fCartographer3 - Waypoints|r is currently disabled."])
		return
	end
	local tokens = { (", "):split(text) }
	
	for i = #tokens, 1, -1 do
		local v = tokens[i]
		if v:trim() == "" then
			table.remove(tokens, i)
		end
	end
	
	if not tokens[1] then
		tokens[1] = ""
	end
	
	local first = tokens[1]:lower()
	
	if first == "clear" then
		Cartographer3_Waypoints.ClearWaypoint()
	elseif first ~= "" then
		local num = 0
		for i, v in ipairs(tokens) do
			if tonumber(v) then
				num = i - 1
				break
			end
		end
		
		local zone = nil
		if num > 0 then
			zone = table.concat(tokens, " ", 1, num)
		end
		
		local x, y = tokens[num + 1], tokens[num + 2]
		
		local name = table.concat(tokens, " ", num + 3)
		if name == "" then
			name = nil
		end
		
		local texture
		if zone then
			if not zoneList then
				zoneList = {}
				for texture, zoneData in pairs(Cartographer3.Data.ZONE_DATA) do
					zoneList[zoneData.localizedName:lower():gsub("[%L]", "")] = texture
				end
			end
			local lowerZone = zone:lower():gsub("[%L]", "")
			for name, texture in pairs(zoneList) do
				if name:match(lowerZone) then
					possibleMatches[#possibleMatches+1] = texture
				end
			end
			if #possibleMatches > 1 then
				local names = {}
				for i, texture in ipairs(possibleMatches) do
					names[#names+1] = Cartographer3.Data.ZONE_DATA[texture].localizedName
				end
				table.sort(names)
				
				DEFAULT_CHAT_FRAME:AddMessage(L["Found multiple matches for zone '%s': %s"]:format(zone, table.concat(names, ", ")))
				for i = 1, #possibleMatches do
					possibleMatches[i] = nil
				end
				return
			end
			texture = possibleMatches[1]
			possibleMatches[1] = nil
			
			if not texture then
				DEFAULT_CHAT_FRAME:AddMessage(L["No match was found for zone '%s'"]:format(zone))
				return
			end
		end
		
		x, y = tonumber(x), tonumber(y)
		if not x or not y then
			printUsage()
			return
		end
		
		x, y = x / 100, y / 100
		
		if not texture then
			texture = GetMapInfo()
		end
		
		Cartographer3_Waypoints.SetWaypoint(texture, x, y, name)
	else
		printUsage()
	end
end
if WotLK then
	local oldHandler = _G.SlashCmdList.CARTOGRAPHERTHREE_WAYPOINTS
	function _G.SlashCmdList.CARTOGRAPHERTHREE_WAYPOINTS(chatFrame, text)
		oldHandler(text)
	end
end

local function ADDON_LOADED(event, addon)
	if not _G.QuestHelper then
		return
	end
	Cartographer3_Utils.RemoveEventListener("ADDON_LOADED", ADDON_LOADED)
	
	QuestHelper:AddWaypointCallback(function(c, z, x, y, desc)
		if not db.questHelper then
			return
		end
		if c then
			Cartographer3_Waypoints.SetWaypoint(c, z, x, y, desc, "QuestHelper")
		end
	end)
end
Cartographer3.Utils.AddEventListener("ADDON_LOADED", ADDON_LOADED)
